/* File: PhylogeneticDistance.h */

#ifndef __PhylogeneticDistance_h__
#define __PhylogeneticDistance_h__

//#define _PRINT_COMPARISON_NUMBER_


namespace BIOS {


struct CMP_METHOD
{
	std::string long_name;
	std::string short_name;	
};


  /************************/
  /* PhylogeneticDistance DEFINITION */
  /************************/


  /**
     @memo PhylogeneticDistance

     @doc
     Definition:

     Class used to compute phylogenetic distance between or within haplotype groups

     With this purpose, the data structures required is:
 
     FreqAndKeyVector* haplotypeList

     where  FreqAndKeyVector is a Vector composed of FreqAndVector elements, defined as HeteroPair<double, long long int> elements (see HeteroPair in /commonc++/HeteroPair.h, FreqAndKeyVector in /commonc++/Container.h

    The definition of those datatypes must be placed in the cpp file of the program in which this class will be used. Check GetPhylogeneticDistances.cpp for an example.

     @author Jose Moreno
     @version 1.0
  */
 
  class	PhylogeneticDistance {

  private:
    //! Set of haplotipes. The distance will be calculated between pairs of haplotypes.
    FreqAndKeyVector* hapList;
    //! Used in the process of extracting the haplotypes' value
    MultidimensionalEmptyTable<int>* haplotypeArray;



      /**
       * Calculates the distance between a pair of haplotypes.
       * It compares the pair of allele values for each marker. Therefore, the haplotypes must have the same length.
       * For each different position the value 1 is added to the distance.
       * @param haplotype1 First haplotype in the pair.
       * @param haplotype2 Second haplotype in the pair.
       * @return 0 if the pair is identical. Greater than 0 if the pair has differences. -1 If there is an error.
       */    
     float DistanceOfPairByDiffPos(intList* haplotype1, intList* haplotype2);

      /**
       * Calculates the distance between a pair of haplotypes.
       * @param haplotype1 First haplotype in the pair.
       * @param haplotype2 Second haplotype in the pair.
       * @return 0 if the pair is identical. 1 if the pair has differences. -1 If there is an error.
       */    
     float DistanceOfPairByDiffBinary(intList* haplotype1, intList* haplotype2);

      /**
       * Calculates the distance between a pair of haplotypes.
       * The method counts the number of switches between the pair.
       * @param haplotype1 First haplotype in the pair.
       * @param haplotype2 Second haplotype in the pair.
       * @return 0 if the pair is identical. Greater than 0 if the pair has differences. -1 If there is an error.
       */    
     float DistanceOfPairByRecomb(intList* haplotype1, intList* haplotype2);

      /**
       * Calculates the distance between each pair of haplotypes given a pointer to a function which calculates de distance between a pair of haplotypes.
       * It compares each pair of haplotypes found in the internal list of haplotypes.
       * @param m Pointer to a function accounted for the calculation of the distance between each pair.
       * @return A value greater than cero if there are differences between the haplotypes.
       */    
    float getWithinDistance( float (PhylogeneticDistance::*m)(intList*, intList*) );

    /**
    *   Iterates the given set by choosing all possible different pairs of haplotypes. Calculates the distance
       * @param m Pointer to a function accounted for the calculation of the distance between each pair.
       * @param hapList List of haplotypes.
       * @return A value greater than cero if there are differences between the haplotypes.
    */
    float calcByIterAll( float (PhylogeneticDistance::*m)(intList*, intList*), FreqAndKeyVector* hapList );

    /**
    *   Iterates by pairing up the most frequent haplotype with each of the remaining haplotypes.
       * @param m Pointer to a function accounted for the calculation of the distance between each pair.
       * @param hapList List of haplotypes.
       * @return A value greater than cero if there are differences between the haplotypes.
    */
    float calcByIterMostFreq( float (PhylogeneticDistance::*m)(intList*, intList*), FreqAndKeyVector* hapList );

    /**
    *   Returns the position of the most frequent haplotype in the list given
    * @param hapList List of haplotypes
    * @return Position in the list of the most frequent haplotype, starting at 0. 
    */
    int  get_most_freq_haplotype( FreqAndKeyVector* hapList );

    /*********************************************************************/
    /***     ASSOCIATED FUNCTIONS     ***/
    /*********************************************************************/

  public:


	static string METHOD_NAME_LONG [];
	static string METHOD_NAME_SHORT [];

	static string ITER_NAME_LONG [];
	static string ITER_NAME_SHORT [];



     //! Total number of possible methods
//	int TOTAL_NUMBER_OF_METHODS;
    static int TOTAL_NUMBER_OF_METHODS;
    //! Current method to use for calculating the distance (type of method)
    int method ;
     //! Total number of possible iterators
    const static int TOTAL_NUMBER_OF_ITERATORS = 2;


     // Types of distance methods
    //! Distance calculated by counting mutations
    const static int METHOD_TYPE_COUNT_MUT = 0; 
    //! Distance calculated by couting recombinations
    const static int METHOD_TYPE_COUNT_RECOMB = 1;
    //! Distance calculated by counting 0 when haplotypes are the same and 1 when they are different.
    const static int METHOD_TYPE_COUNT_BINARY = 2;



    //! Specifies the current way of selecting the pairs to compare.
    int iterator ;
    //! Create all different possible pairs of haplotypes
    const static int ITER_TYPE_ALL = 0;  
    //! Create all pairs between the most frequent haplotype and the rest
    const static int ITER_TYPE_MOST_FREQ = 1;  


      /**
       * Constructor
       * @param hapList List of haplotypes.
       * @param haplotypeArray auxiliar variable to extract haplotypes.
       */
    PhylogeneticDistance(FreqAndKeyVector* hapList, MultidimensionalEmptyTable<int>* haplotypeArray); 


      /**
       * Constructor
       * @param haplotypeArray auxiliar variable to extract haplotypes.
       */
    PhylogeneticDistance(MultidimensionalEmptyTable<int>* haplotypeArray); 



    ~PhylogeneticDistance();

    /**
     * Calculates the distance between the haplotypes inside the class and hapList2 using the method idicated.
     * @param method Function used to calculate the distance.
     * @param hapList2 List of haplotipes to calculate the distance to.
     * @return Value of global distance between the two sets.
     */
    float getBetweenDistance(int method, FreqAndKeyVector* hapList2); // it computes distance between hapList and hapList2

    /**
    * Calculates the distance between the haplotypes in the list given. The settings for the method to use must be specified by the corresponding methods. Check PhylogeneticDistance::method and PhylogeneticDistance::iterator
    * @param haplist Lista de haplotipos a comparar
    *  @return Distance between the elements in the set.
    */
    float calculate(FreqAndKeyVector* hapList); 

    /**
     * Calculates the distance between the haplotypes inside the class and hapList2.
     * The method counts the number of mutations between each pair of haplotypes. That is the number of different positions.
     * @param hapList2 List of haplotipes to calculate the distance to.
     * @return Value of global distance between the two sets. 
     */
    float getBetweenDistanceCountMut(FreqAndKeyVector* hapList2);

    /**
     * Calculates the distance between the haplotypes inside the class and hapList2.
     * The method counts the number of recombinations between each pair of haplotypes. 
     * @param hapList2 List of haplotipes to calculate the distance to.
     * @return Value of global distance between the two sets. 
     */
    float getBetweenDistanceCountRecomb(FreqAndKeyVector* hapList2);


    /**
     * Calculates the distance between the haplotypes inside the class.
     * @param method Function used to calculte the distance.
     * @return Value of global distance between the two sets.     
     */
    float getWithinDistance(int method); 

    /**
     * Calculates the distance between the haplotypes inside the class.
     * The method counts the number of mutations between each pair of haplotypes. That is the number of different positions.
     * @param hapList2 List of haplotipes to calculate the distance to.
     * @return Value of global distance between the two sets. 
     */
    //float getWithinDistanceCountMut();

    /**
     * Calculates the distance between the haplotypes inside the class.
     * The method counts the number of recombinations between each pair of haplotypes. 
     * @param hapList2 List of haplotipes to calculate the distance to.
     * @return Value of global distance between the two sets. 
     */
    //float getWithinDistanceCountRecomb();

    /**
     * Returns the list of haplotypes.
     * @return Haplotype list.
     */
    FreqAndKeyVector* getHapList();

    /**
     * Long names for the methods
     */
    static string getDistanceAlgorithmLongName(int method){

	/*
        switch(method){
            case METHOD_TYPE_COUNT_MUT:
                return "Method: Count mutations";
                break;
            case METHOD_TYPE_COUNT_RECOMB:
                return "Method: Count recombinations";
                break;
            case METHOD_TYPE_COUNT_BINARY:
                return "Method: Binary";
                break;
            default:
                return "Method unknown";
                break;
        }
*/
	//return methods_vector[method].long_name;
	//cout << endl << "i: " << method << endl;
	//return "aa";
	return PhylogeneticDistance::METHOD_NAME_LONG[method];
    	    
	};

    /**
     * Short names for the methods
     */
    static string getDistanceAlgorithmShortName(int method){
	return PhylogeneticDistance::METHOD_NAME_SHORT[method];
	}

    /**
     * Long names for the iterators
     */
    static string getIteratorLongName(int iterator){


		return PhylogeneticDistance::ITER_NAME_LONG[iterator];
    	    
	};

    /**
     * Long names for the iterators
     */
    static string getIteratorShortName(int iterator){
		return PhylogeneticDistance::ITER_NAME_SHORT[iterator];
    	    
	};

   
    /**
     * Returns the value of one haplotype inside the class.
     * @param key key of the haplotype in the internal list. The key must be obtained from the second element in the pair at the desired position in the array. For example, the key for the ith element is "hapList->getElement(i)->getSecond();"
     * @return List of integers representing the allele values for the markers of the haplotype.
     */
    intList* getHaplotype(long long int key);

    /**
     * Returns the value of one haplotype inside the class.
     * @param listPosition Position in the list. More friendly approach than using key to extract the haplotype information.
     * @return List of integers representing the allele values for the markers of the haplotype.
     */
    intList* getHaplotypeAt(long int listPosition, FreqAndKeyVector* hapList);


    

  };  // End of class PhylogeneticDistance

 



  ostream& operator<<(ostream& out, PhylogeneticDistance& pm)
    {
      if (pm.getHapList()!=NULL)
	out << *pm.getHapList() <<"\n";


      return out;
    };


  /*______________________________________________________*/



};  // End of Namespace

#endif

/* End of file: PhylogeneticDistance.h */




