#ifndef __HaplotypeTUCountsVector_h__
#define __HaplotypeTUCountsVector_h__

namespace BIOS {
	


typedef Vector<HaplotypeTUCounts*>::Class HaplotypeTUCountsList;

	/**
	* List of haplotypes
	*/
	class HaplotypeTUCountsVector : public HaplotypeTUCountsList {
//class HaplotypeTUCountsVector : public std::vector<Haplotype*>{
		private:
		
		public:
			/**
			* Constructor
			*/			
			HaplotypeTUCountsVector(){};

			/**
			* Destructor. It frees only the pointers, no the objects pointed.
			*/	
			~HaplotypeTUCountsVector();

			/** 
			* Free the memory used by the objects pointed (the haplotypes).
			*/
			void free();

			/**
			*	Deletes the current vector, freeing the memory.
			*	Calls free and the destructor.
			*	@see free();
			*/
			static void Delete(HaplotypeTUCountsVector * & hv);

			/**
			*	Deletes the current vector, freeing the memory.
			*	Calls free and the destructor.
			*	@see free();
			*/
			void Delete();

			/**
			* Copy constructor
			* Used to copy the whole vector, not only the pointers.
			*/
			HaplotypeTUCountsVector(const HaplotypeTUCountsVector & v);

   
			HaplotypeTUCountsVector* clone();
			
			/**
			* Creates a new vector of haplotypes.
			* The frequency of each haplotype is 1.
			* The haplotypes are duplicated.
			* @return new vector created.
			*/
			HaplotypeTUCountsVector* expand();

			/**
			* Insert the elements of the given vector into this one.
			* Not only the pointers, but new copies of the elements
			* @param v Haplotype vector from which to read the elements to insert.
			*/	
			void insertClone(HaplotypeTUCountsVector * v);

			/**
			*	Removes one element from the vector. Also frees the memory of the deleted object 
			*/
			void remove(int index);

			/**
			* Insert the elements of the given vector into this one.
			* @param hv Haplotype vector from which to read the elements to insert.
			*/	
			//void insertVector(HaplotypeTUCountsVector hv);

			/**
			* Returns the position of the first different marker in the haplotype vector
			*/
			int firstDifferentMarker();

			/**
			*	 Returns the position of the first different marker in the haplotype vector after last_marker
			*	@pram last_marker Position to start looking for differences.
			*	@param toLeft Indicates the order in which to process the splitting process. When toLeft is true, the alleles are compared from left to right.
			*/
			int nextDifferentMarker(int last_marker, bool toLeft=false);

			/**
			* Z statistic 
			* @param p Proportion of T/U haplotypes in the whole population
			*/
			double getZstatistic( double p );
			
			/**
			*	Get proportion of transmitted haplotypes without having into account frecuencies.
			*/
			float getTransmittedCountProportion();

			/**
			*	Number of transmitted haplotypes. Without frecuencies 
			*/
			int getTransmittedCount(){};

			/**
			*	Sums the frequencies of the transmitted haplotypes
			*/
			double getTransmittedFreqSum();

			/**
			*  Sum of frequencies of the sample 
			*/
			double getFreqSum();

			/**
			*	Sums the frequencies of the non transmitted haplotypes
			*/
			double getNonTransmittedFreqSum();

			/**
			*	Splits the haplotypes vector into two subsets by the next different marker to 'last_marker'.
			*	@param[out] v1 First of the vectors created. Haplotypes with 0 at the different marker.
			*	@param[out] v2 One of the vectors created. Haplotypes with 1 at the different marker.
			*	@param[in] last_marker The next different marker will be search after last_marker
			*	@param toLeft Indicates the order in which to process the splitting process. When toLeft is true, the alleles are compared from left to right.
			*	@return Position by which the vector was splitted
			*/
			int split(HaplotypeTUCountsVector * & v1, HaplotypeTUCountsVector * &v2, int last_marker, bool toLeft=false);


			/**
			*	Splits the haplotypes vector into two subsets by the position indicated. The new subsets will have the same number of haplotypes. 
				The size of the new haplotypes will be determined by the position of chopping.
			*	@param[out] v1 First of the vectors created. Haplotypes with 0 at the different marker.
			*	@param[out] v2 One of the vectors created. Haplotypes with 1 at the different marker.
			*	@param[in] pos Position in which to split the haplotypes
			*/
			void chop(HaplotypeTUCountsVector * & v1, HaplotypeTUCountsVector * &v2, int pos);
		
			/**
			*	Chops the vector by the middle
			*	@see HaplotypeTUCountsVector::chop()
			*	@param[out] v1 First of the vectors created. Haplotypes with 0 at the different marker.
			*	@param[out] v2 One of the vectors created. Haplotypes with 1 at the different marker.
			*/
			void chopInHalf(HaplotypeTUCountsVector * & v1, HaplotypeTUCountsVector * &v2);

			/**
			*	Creates a new vector whose elemenets have only the positions indicated.
			*	@param[in] pos Vector with the positions to extract
			*	@param[in] length Number of elements in vector position
			*	@return New HaplotypeTUCountsVector with the positions indicated. Duplicated elments will be collapsed.
			*	@see HaplotypeTUCountsVector::collapse()
			*/
			HaplotypeTUCountsVector * filter(int *pos, int length);

			/**
			*	Groups properties for elements with the same internal haplotype. That is, duplicated haplotypes.
			*/
			void collapse();
			
			/**
			* Copy frequencies from the vector given.
			* The frequencies of the haplotypes in this vector are changed.
			*/
			void copyFrequencies(HaplotypeTUCountsVector * & v1);


			/**
			*	Compares two vectors
			*	@return true if vectors are equal
			*/
			bool operator==(HaplotypeTUCountsVector & h);

			/**
			*	Compares two vectors
			*	@return true if vectors are different
			*/
			bool operator!=(HaplotypeTUCountsVector & h);

			/**
			* Prints the haplotype vector. Firstly presents the vector size, and then the haplotypes themselves.
			* @param output Output stream
			* @param hv Vector of halotypes to print
			*/
			friend ostream& operator<<(ostream& output, const HaplotypeTUCountsVector& hv){
	
				output << "Size: " <<  hv.size() << endl;
	
				for (int i=0; i< hv.size(); i++)
					output << *(hv[i]) <<endl;

				return output;

			}		
void print();
	
	};



	typedef Vector<HaplotypeTUCountsVector*>::Class VectorOfHaplotypeTUCountsVectors;

	/**
	*	A partition of haplotypes. 
	*	A vector of haplotype vectors.
	*	In practice, this is a partition of the TreeDT algorithm;	
	*/
	class PartitionHaplotypeTUCountsVector : public VectorOfHaplotypeTUCountsVectors{
		private:


		public:
			//! List of marker by which each subset was generated in the partition
			std::vector<int> splitMarkers;

			/**	
			*	Constructor
			*/
			PartitionHaplotypeTUCountsVector();

			/**	
			*	Destructor. 
			*	Deletes also the objects pointed in the vector.
			*/
			~PartitionHaplotypeTUCountsVector();

   PartitionHaplotypeTUCountsVector* clone();

			/** 
			* Free the memory used by the haplotypes contained in the subsets (the haplotypes).
			*/
			void free();

			/**
			*	Calculates the z statistic of the partition as the sum of the z statistics of the subsets (HaplotypeTUCountsVector)
			*	@see HaplotypeTUCountsVector::getZstatistic()
			*	@param p Proportion of T/U haplotypes in the whole population
			*	@return z statistic
			*/
			double getZstatistic(double p);


			/**
			*	Inserts the vector given as a subset of the partition. 
			*	@param hv Haplotype vector to insert 
			*	@param splitMarker Marker by which the partition was made	
			*/
			void insertSubset(HaplotypeTUCountsVector *hv, int splitMarker=-1);

			/**
			*	Inserts the copy of the vector given as a subset of the partition. 
			*	@param hv Haplotype vector to insert 
			*	@param splitMarker Marker by which the partition was made	
			*/
			void insertSubsetCopy(HaplotypeTUCountsVector *, int splitMarker=-1);

			//TDTtable * createTable();

			/**
			*	print the partition 
			*/
			friend ostream& operator<<(ostream& output, const PartitionHaplotypeTUCountsVector& hv){

				output << " - Number of sub-sets: " <<  hv.size() << endl << endl;

				for (int i=0; i< hv.size(); i++)
					output << "subset " << i  << ") "<< *(hv[i]) <<endl;

				return output;

			};

void print();

	};
	



};

#endif
