// MLC++ - Machine Learning Library in -*- C++ -*-
// See Descrip.txt for terms and conditions relating to use and distribution.

// This is an include file.  For a full description of the class and
// functions, see the file name with a "c" suffix.

#ifndef Array_h
#define Array_h

template <class Element>
class Array {
   NO_COPY_CTOR(Array);

   // Member data (also see protected data)
   Bool owner;
protected:   // in case we want to derive a DynamicArray from Array.
   // Protected member data
   int base;
   int arraySize; 
   Element *elements;   // the name array conflicts with LEDA
   // Protected methods
   void alloc_array(int baseStart, int size); 
   void copy(const Array& source); // used by copy constructor
   #ifdef PC_MSVC_AMBIG
   Bool equal_given_that_bounds_are_equal(const Array<Element>& rhs) const;
   #endif
public:
   Array(const Array<Element>& source, CtorDummy);
   Array(int baseStart, int size);
   Array(int size);
   Array(int baseStart, int size, const Element& initialValue); 
   Array(Element *carray, int length);
   void truncate(int size);
   virtual ~Array() { if (owner) delete[] elements; }  
   Array& operator=(const Array& source);
   Element *get_elements() { return elements; } 
   void init_values(const Element& initalValue);
   int high() const { return base + arraySize - 1; } 
   int low()  const { return base; }
   int size() const { return arraySize; }
   // read, write, display need MLCstream routines for reading/writing
   //  the object.
   void read_bin(MLCIStream& str);
   void write_bin(MLCOStream& str);
   void display(MLCOStream& stream = Mcout) const;
   Array<Element>& operator +=(const Array<Element>& item);
   Array<Element>& operator -=(const Array<Element>& item);


/***************************************************************************
    Description :  The next two functions return a reference and a const
                     reference, respectively, to the correct element
		     referenced from a ZERO BASE.
		   The two functions after return the element with respect
		     to the true base.
    Comments    :  Out of bounds reference is a fatal error.
***************************************************************************/
   // CFront 3 chokes if these are moved outside the class
   // and defined as inline (it generates static function undefined).
   Element& index(int i) {
      DBG(if (i < 0 || i >= arraySize)
	  err << "Array<>::index: reference " << i
	      << " is out of bounds (0.." << arraySize - 1
	      << ')' << fatal_error
	  ); 
      return elements[i];
   }

   const Element& index(int i) const {
      DBG(if (i < 0 || i >= arraySize)
	  err << "Array<>::index() const:  reference "  << i
	      << " is out of bounds (0.."
	      <<  arraySize - 1 << ')' << fatal_error
	  ); 
  
      return elements[i];
   }

   Element& operator[](int i) {
      DBG(if (i < base || i > base + arraySize - 1)
	  err << "Array<>::operator[]: reference " << i
	      << " is out of bounds (" << base 
	      << ".." << high() << ')' << fatal_error
	  ); 
      return elements[i - base];
   }

   inline const Element& operator[](int i) const {
      DBG(if (i < base || i > base + arraySize - 1)
	  err << "Array<>::operator[] const: reference "  << i
	      << " is out of bounds (" << base << ".." << high()
	      << ')' << fatal_error
	  ); 
  
      return elements[i - base];
   }

   // The following two methods return the same data as the index()
   //   methods--but the fast_index() methods do NOT do ANY error checking.
   //   They are provided for speed--not safety.
   Element& fast_index(int i) {return elements[i];}
   const Element& fast_index(int i) const {return elements[i];}
   // sort, min/max need operator<
   void sort();
   const Element& max(int& idx) const;  // idx is zero based index to array
   const Element& max() const;
   const Element& min(int& idx) const;  // idx is zero based index to array
   const Element& min() const;
   void  min_max(const Element& min, const Element& max) const;
   Element sum() const; // Sum the elements of the array.

   int find(const Element& key, int startPos=0) const;
   
   int num_element(const Element& key) const;
   MString get_element_indexes(const Element& key) const;

   #ifdef PC_MSVC_AMBIG
   Bool bounds_are_equal(const Array<Element>& rhs) const;
   Bool operator==(const Array<Element>& rhs) const;
   Bool operator!=(const Array<Element>& rhs) const;
   #endif
};

template <class Element>
class PtrArray: public Array<Element> {
   NO_COPY_CTOR(PtrArray);
protected:
   #ifdef PC_MSVC_AMBIG
   Bool equal_given_that_bounds_are_equal(const PtrArray<Element>& rhs) const;
   #endif
public:
   // Operator= aborts by default.  Use Array::operator= if the class
   // has a well defined operator= for its element (rarely because it's
   // usually class* that's used in the template)
   Array<Element>& operator= (const Array<Element>& source);
   PtrArray(int baseStart, int size, Bool init=TRUE);
   PtrArray(int size); // initializes to NULL
   ~PtrArray();          
   void display(MLCOStream& stream = Mcout) const;
   void sort();

   #ifdef PC_MSVC_AMBIG
   Bool bounds_are_equal(const PtrArray<Element>& rhs) const;
   Bool operator==(const PtrArray<Element>& rhs) const;
   Bool operator!=(const PtrArray<Element>& rhs) const;
   #endif
};

#ifndef PC_MSVC_AMBIG
// Template functions
//@@ Under MSVC, these functions are declared directly inside their
//@@ respective classes instead of outside.  We do this because MSVC
//@@ incorrectly complains about these functions being ambiguous
//@@ (PtrArray versions vs. Array versions)
//@@ The DECLARE_DISPLAY is completely removed for Arrays because it
//@@ causes a large number of compiler bugs under MSVC if left in.
template <class Element> 
Bool equal_given_that_bounds_are_equal(const Array<Element>& lhs,
				       const Array<Element>& rhs);
template <class Element> 
Bool bounds_are_equal(const Array<Element>& lhs,
		      const Array<Element>& rhs);
template <class Element> 
Bool operator==(const Array<Element>& lhs, const Array<Element>& rhs);

template <class Element> 
Bool operator!=(const Array<Element>& lhs, const Array<Element>& rhs);


template <class Element> 
Bool equal_given_that_bounds_are_equal(const PtrArray<Element>& lhs,
				       const PtrArray<Element>& rhs);
template <class Element> 
Bool bounds_are_equal(const PtrArray<Element>& lhs,
		      const PtrArray<Element>& rhs);
template <class Element> 
Bool operator==(const PtrArray<Element>& lhs, const PtrArray<Element>& rhs);

template <class Element> 
Bool operator!=(const PtrArray<Element>& lhs, const PtrArray<Element>& rhs);

template <class Element>
DECLARE_DISPLAY(Array<Element>)
#endif

template <class Element>
DECLARE_DISPLAY(PtrArray<Element>)

#ifdef COMPILE_TIME_TEMPLATES
#include "Array.ct"
#endif
   
#endif


