#ifndef Dv2d_CPP
#define Dv2d_CPP 1

namespace BIOS {

dv2d dv2d::operator+(dv2d const & mat2)
// Matrix addition: SUM=A+B
  {
  dv2d sum(high,wide);
  for (int i=0;i<high;++i)
    for (int j=0;j<wide;++j)
      sum.rv[i][j]=rv[i][j]+mat2.rv[i][j];
  return sum;
  }

/*___________________________________________________________*/

dv2d dv2d::operator*(dv2d const & mat2)
// Matrix multiplication: SUM=A*B
  {
  dv2d sum(high,mat2.wide);
  for (int i=0;i<sum.high;++i)
    {
    for (int j=0;j<sum.wide;++j)
     {
     double temp=0;
       {
       for (int k=0;k<wide;++k)
         temp+=rv[i][k]*mat2.rv[k][j];
       }
     sum.rv[i][j]=temp;
     }
    }
  return sum;
  }

/*___________________________________________________________*/


dv2d& dv2d::operator=(dv2d const & othermat)
// Matrix assignment: A=B
 {
 if (wide!=othermat.wide || high!=othermat.high)
    resize(othermat.high,othermat.wide);
 if (size())
    for (int i=0;i<high;++i)
      memmove(rv[i],othermat.rv[i],wide*sizeof(double));
 return *this;
 }



/*___________________________________________________________*/

dv2d::dv2d(int h,int w)
// Constructor, h rows and w columns
 {
 high=wide=0;
 rv=0;
 resize(h,w);
 }


/*___________________________________________________________*/

dv2d::dv2d(dv2d const &othermat)
// Copy constructor
 {
 high=wide=0;
 rv=0;
 if (resize(othermat.high,othermat.wide))
   operator=(othermat);
 }

/*___________________________________________________________*/

dv2d* dv2d::clone()
// Copy constructor
 {
return new dv2d(*this);
 }

/*___________________________________________________________*/

dv2d::~dv2d()
// Destructor
 {
 if (rv)
   {
   for (int i=0;i<high;++i)
    delete rv[i];
   delete rv;
   }
 }


/*___________________________________________________________*/

int dv2d::resize(int h,int w)
// Carefully allocate and deallocate memory as required
// It might be more efficient not to resize if more
// than enough memory is already allocated, but that
// would involve additional house-keeping
 {
 int i;
 if (h==high && w==wide) return 1;
 if (rv)
   {
   for (i=0;i<high;++i)
     delete rv[i];
   delete rv;
   }
 max_h=h;
 high=h;
 wide=w;
 if (h==0 && w==0) return 1; // a mistake for only one to be 0
 rv=new DBLPOINTER[max_h];
 if (rv==NULL)
  {
  dvec_error("Out of memory in dv2d (1)");
  high=wide=0;
  return 0;
  }
 for (i=0;i<h;++i)
  if ((rv[i]=new double[w])==NULL)
  {
  dvec_error("Out of memory in dv2d (2)");
  for (;i;--i) delete rv[i-1];
  delete rv;
  high=wide=max_h=0;
  rv=0;
  return 0;
  }
 return 1;
 }



/*___________________________________________________________*/

void dv2d::normal(int code)
// Normalise columns
// Might want to do this before principal components analysis
// 1 to subtract means, 2 to divide by sd
 {
 int i,j;
 doublevec mean(wide),sd(wide);
 for (i=0;i<wide;++i)
   mean[i]=sd[i]=0.0;
 for (i=0;i<high;++i)
    for (j=0;j<wide;++j)
      {
      double x;
      x=rv[i][j];
      mean[j]+=x;
      sd[j]+=x*x;
      }
 for (i=0;i<wide;++i)
    {
    mean[i]/=high;
    sd[i]=sqrt(sd[i]/high-mean[i]*mean[i]);
    }
 if (code==1||code==2)
  for (i=0;i<high;++i)
    for (j=0;j<wide;++j)
       {
       rv[i][j]-=mean[j];
       if (code==2) rv[i][j]/=sd[j];
       }
 }


#if 0
dv2d dsvd::svbksb(dv2d& b)
// Solve for X in AX=b where A is original matrix before decomposition
// Adapted from Numerical recipes in C, Press et al
   {
   dv2d x(W.wide,1);  // as high as W is wide
   int jj,j,i;
   double s;
   setU(); setV(); // could rewrite this later so won't need them
   int n=U.wide,m=U.high;
   doublevec tmp(n+1);
   for (j=0;j<n;j++) {
      s=0.0;
      if (W[j][j]) {
         for (i=0;i<m;i++) s += U[i][j]*b[i][0];
         s /= W[j][j];
      }
      tmp[j]=s;
   }
   for (j=0;j<n;j++) {
      s=0.0;
      for (jj=0;jj<n;jj++) s += V[j][jj]*tmp[jj];
      x[j][0]=s;
   }
return x;
}
#endif


/*___________________________________________________________*/

dv2d dv2d::inv()
// Return inverse of matrix, SUM=A.inv()
{
dv2d sum(1,1);
if (high==1 && wide==1) { sum[0][0]=1/rv[0][0]; return sum; }
dsvd s(high,wide);
if (s.dcmp(*this)==0) throw ZeroValue("dv2d dv2d::inv()");
s.setWinv();
s.setV();
sum=s.V*s.Winv*s.UT;
return sum;
}
/*___________________________________________________________*/

bool dv2d::zeroInverse()
// Return inverse of matrix, SUM=A.inv()
{
if (high==1 && wide==1) { return false; }
dsvd s(high,wide);
if (s.dcmp(*this)==0) return true;
return false;
}


/*___________________________________________________________*/

dv2d dv2d::transpose()
// Matrix transposition SUM=A.transpose()
 {
 dv2d trans(wide,high);
 if (trans.size())
  for (int i=0;i<high;++i)
   for (int j=0;j<wide;++j)
     trans[j][i]=rv[i][j];
 return trans;
 }

/*___________________________________________________________*/

dv2d& dv2d::operator+=(dv2d const & mat2)
// Matrix addition: A=A+B, written A+=B
  {
  for (int i=0;i<high;++i)
    for (int j=0;j<wide;++j)
        rv[i][j]+=mat2.rv[i][j];
  return *this;
  }

/*___________________________________________________________*/

dv2d dv2d::operator-(dv2d const & mat2)
// Matrix subtraction: A=A-B, written A-=B
  {
  dv2d sum(high,wide);
  for (int i=0;i<high;++i)
    for (int j=0;j<wide;++j)
        sum[i][j]=rv[i][j]-mat2.rv[i][j];
  return sum;
  }

/*___________________________________________________________*/

dv2d& dv2d::operator*=(dv2d const & mat2)
// Matrix multiplication: A=A*B, written A*=B
{
*this=(*this)*mat2;  // no shortcut to multiplying
return *this;
}


/*___________________________________________________________*/

dv2d dv2d::operator*(double f)
// Scale whole matrix by a constant: SUM=A*f
  {
  dv2d sum(high,wide);
  for (int i=0;i<high;++i)
    for (int j=0;j<wide;++j)
       sum.rv[i][j]=rv[i][j]*f;
  return sum;
  }


/*___________________________________________________________*/

dv2d& dv2d::operator*=(double f)
// Scale whole matrix by a constant: A=A*f, written A*=f
{
int i,j;
for (i=0;i<high;++i)
   for (j=0;j<wide;++i)
        rv[i][j]*=f;
return *this;
}


/*___________________________________________________________*/


void dv2d::fscanf(FILE *fp)
// Read in matrix from file
 {
 for (int i=0;i<high;++i)
   for (int k=0;k<wide;++k) ::fscanf(fp,"%lf",&rv[i][k]);
 }
 
 /*___________________________________________________________*/
 
  ostream& operator<<(ostream& out, dv2d& l)
  {
      for (int j=0; j<l.get_width(); j++) out << "------";
 
  for (int i=0; i<l.get_height(); i++)
  {
  out <<"\n|";
    for (int j=0; j<l.get_width(); j++)
        out << setw(5) <<  l.vec()[i][j] <<"|";
  //out << l.vec()[i,j] <<"|\t|";
  out <<"\n";
   for (int j=0; j<l.get_width(); j++)
   out  << "------";
  }
     out <<"\n";
  return out;
  }

}

#endif
