#ifndef __TreeDTMeasure_h__
#define __TreeDTMeasure_h__

namespace BIOS {
	
	/**
		@memo TreeDTMeasure
		@doc
		Definition:

		TreeDT algorithm inspired in Sevon et al. "TreeDT: Tree Pattern Mining for Gene Mapping" (2006)

		@author Jose J. Moreno
		@version 1.0
	*/	
	
	//! Z statistic value for a concrete k, for each permutation. For example, the 10(npermu) z statistics for the level 0.
	//! Plus the z statistic observed as the last component.
	double *tmp_score;	
	
	int cmp_sort(const void *, const void *);

struct NodeTreeDT {
	struct NodeTreeDT *next, *father;
	struct ListNodeTreeDT *first_child, *last_child;
	long na, nc, nchr, nm;
	//min: In the tree, the minimum index of the haplotypes in the subtree
	//max: In the tree, the maximum index of the haplotypes in the subtree
	long min, max; /* in sorted list */ // min and max represent a haplotype index (starting at 0)
	long chromo; // Index of haplotype. starting at 0
	long depth; // Depth of the node in the tree
	long shared_length;
	//loc: position of the node in the chromosome
	double mean_shared, loc;

friend ostream& operator<<(ostream& out, NodeTreeDT& nt)
{
out << "na: " << nt.na << "\n"; 
out << "nc: " << nt.nc << "\n"; 
out << "nchr: " << nt.nchr << "\n"; 
out << "nm: " << nt.nm << "\n"; 
out << "min: " << nt.min << "\n"; 
out << "max: " << nt.max << "\n"; 
out << "chromo: " << nt.chromo << "\n"; 
out << "depth: " << nt.depth << "\n"; 
out << "shared_length: " << nt.shared_length << "\n"; 
out << "mean_shared: " << nt.mean_shared << "\n"; 
out << "loc: " << nt.loc << "\n"; 
}

};

struct ListNodeTreeDT {
	struct NodeTreeDT *ptr;
	struct ListNodeTreeDT *next, *prev;
};
	
// From io.c
#define APOOLSIZE 16
struct AllelePool {
	struct AllelePool *next;
	int cnt;
	int key[APOOLSIZE];
};
#define POOLSIZE 1024
struct NodePool {
	struct NodePool *next;
	struct NodeTreeDT node[POOLSIZE];
};

struct ListNodePool {
	struct ListNodePool *next;
	struct ListNodeTreeDT listnode[POOLSIZE];
};



class TreeDTMeasure: public GroupBasedTDTMeasure{
	  

		TreeDTModel * model;
		
		//! Chooses the implementation used. 0 for the original treedt. 1 for the new one.
		int implementation;
		
	
		//! Copy used for permutations
		//HaplotypeTUCountsVector* copyHaplotypeTUCountsVector;
		
		//! trees on the left to right pass
		vector<TreeOfHaplotypeTUCounts *> vectorOfTrees1;
		
		vector<HaplotypeTUCountsTree *> leftTrees;
		vector<HaplotypeTUCountsTree *> rightTrees;
		
		//! Position with minimum p-value at level 3
		long posMinPlevel3;

		//! trees on the right to left pass
		vector<TreeOfHaplotypeTUCounts *> vectorOfTrees2;
		
		
		//! TDTtable for each marker (left to right). Calculated using the best nodes of the corresponding tree.
		vector<TDTtable<T> > vectorOfTables1;

		//! TDTtables for each marker (left to right). Calculated using the best nodes of the corresponding tree.
		vector<TDTtable<T> > vectorOfTables2;
		
		TDTtable<T> * table2;
		
// 		template <class T>
// 		void copyVector(vector<T *> & original, vector<T *> &copy );
			void copyVector(vector<TreeOfHaplotypeTUCounts *> & original, vector<TreeOfHaplotypeTUCounts *> &copy );
			
			void copyTableVector(vector<TDTtable<T> > & original, vector<TDTtable<T> > &copy );

		
		//! Sum of all T frequencies
		float sumFrequencyT;
		//! Sum of all U frequencies
		float sumFrequencyU;
		
		//! Sum of all frequencies (real number of chromosomes). nchromo is the number of different chromosomes
		float nchromoF;
	  
	  
		//! Vector of haplotypes vector with repetitions and freq=1 per haplotype
		//HaplotypeTUCountsVector* expandedHaplotypeVector;

// From main.c

		const static int MAX_SUBTREES=3;
		const static int NSUBSCORES=MAX_SUBTREES;
		TUCounts* tuCounts;// this should not be here???, we used counts from GenericMeasure

		// mdata: Alleles in the sample. All in one array. The values of mdata are indexes in the AllelePools.
		// status: Array with the status (a/c) for each haplotype. ["a" (affected) and "c" (unaffected/control)]
		// maxallele: Total number of different alleles in each marker. maxallele[0] is the number of different alleles at marker 0.
		char *mdata, *status, *maxallele, *mutation;

		// nchromo: number of different haplotypes (without frequencies)
		//nmarkers: number of markers
		// na: number of haplotypes with state a
		// nc: number of haplotypes with state c
		long nchromo, nmarkers, na, nc, nmuta;
		

		
		long npermu;
		long seed;
		double obs_score1[MAX_SUBTREES];
		// location: For each marker, the position in double (first marker=1.0, second marker=2.0, etc.)
		double correct_locus, *location, current_location;
		long closest_mk, tree_mk;
		int muta_info;
		char mode;
		// Using min p-value across k's if fixed_k = 0
		long fixed_k;

		struct NodeTreeDT *root;


// From io.c
		struct AllelePool **first_apool;
		long nid;
		int dir;		

// From tree.c
		long *sortlist;
		long pos, sortlistbase;
		// root_a: Root node associated with each allele. For example, root_a[2] is the node associated with allele 2.
		struct NodeTreeDT *root_a[256], *last_node[256];
		struct NodeTreeDT *first_free_node;
		struct ListNodeTreeDT *first_free_listnode;
		struct NodePool *first_npool;
		struct ListNodePool *first_lnpool;

// From permu_i.c

	//! P-value for each marker. The statistic to calculate this p-value is the product of the minimum p-values of left and right (found in obs_score2).
	double *obs_pval2;
	double obs_pval1[NSUBSCORES];
	
	//! Minimum p-value across k's. Two p-values for each marker. Given a marker, the z distribution for each k is calculated. Each k gives a p-value and the minimum is stores here.
	double *obs_score2;

	//! Index in the array tmp_score. It is used to sort the permutation values
	long *tmp_ptr;
//akiiiii
	// permu_score1: Z statistic values for left to right pass. For each permutation (total npermu) there are "NSUBSCORES==MAX_SUBTREES" values.
	// permu_score2: 
	double *permu_score1, *permu_score2;
	double *pval1;
	char *permu_status;
	double *min_permu_pval2;
	int phase;

// From score.c
	double tmp_scoretable[MAX_SUBTREES];


		//! to keep track of the inner nodes created 
// 		vector<HaplotypeTUCounts *> treeEmtpyNodes;
		
		typedef Set<HaplotypeTUCounts *>::Class::iterator PNode;

		public:
		double overall_p;
		
		//! 0 for original TreeDT statistic (default). 1 for z statistic squared.
		int statisticUsed;		

		/**
		*	Constructor
		*/		
		//TreeDTMeasure();

		/**
		*	Constructor
		*	@param npermu Number of permutations. By default 10
		*/	

		TreeDTMeasure(SampleGeneticCounts<T>* tuCounts=NULL, int npermu=100, int implementation=1, double minFreq=10, int testMode=0, SampleGeneticCounts<T>** partialTuCountsTraning=NULL, SampleGeneticCounts<T>** partialTuCountsTest=NULL);
		
		
		/**
		 * Copy constructor
		 * @param other Measure to clone
		 */
		TreeDTMeasure(TreeDTMeasure & other);
		

		/**
		*	Destructor
		*/		
		virtual ~TreeDTMeasure();
		
		void setPermutations(int npermu);
		int getPermutations();

		virtual TreeDTMeasure* clone();


		/**
		*	virtual methods to get pval of the measure.
		*	
		*/

		virtual double getStatistic();

		/**
		*	Calculates the p-value of the sample set as the minimum p_k. 
		*	A p_k value is calculated for each k.
		*	@see getPvalueForK()
		*	@return p-value		
		*/

		virtual double getPVal();
		virtual double getPVal_mode0();
		virtual double getPVal_mode2();
		virtual double getPVal1();
		
		TreeDTModel * createModel(HaplotypeTUCountsVector *v);
		
		double getPVal(TreeDTModel *model);		
		
		/**
		* Creates the set of left and right trees (which are stored as class members)
		* @param v Vector of haplotypes from which to create the trees
		*/
		void createTrees();


		virtual string getName();

		virtual stringList* getHeadFile();

		virtual TreeDTMeasure* getNewMeasure(SampleGenericCounts* tuCounts, SampleGenericCounts** training=NULL, SampleGenericCounts** test=NULL);
		
		
		virtual TreeDTMeasure* inferMeasure(SampleGeneticCounts<T>* tuCounts);

		TUCounts* getTUCounts();

		TreeDTMeasure* fromString(string s);

		//friend ostream& operator<<(ostream& out, TreeDTMeasure& l){};
		
		void print(ostream&);


		void setTUCounts(TUCounts* tuCounts);


		/* TreeDT */


		private:
			// From io.c
			void read_data();
			int print_tree(struct NodeTreeDT *n, int level);
			void debug_dump_chromo(long i);
			int original_allele(long m, char c);
			void write_subscores(FILE *f, long m);
			void dispose_data(void);
			void write_power(FILE *f);
			
			void rec_write_tree(FILE *f, struct NodeTreeDT *n, long level, long pos, long shared_length);
			void write_left_tree(FILE *f, struct NodeTreeDT *root, long pos);
			void write_right_tree(FILE *f, struct NodeTreeDT *root, long pos);
			
			void write_sub_pvalues(FILE *f, long m);
			char map_allele(long m, int a);
			
			// From tree.c			
			void init_left_sort(void);
			void init_right_sort(void);
			void init_sort(void);
			struct NodeTreeDT *allocate_node(void);
			void treelink(struct NodeTreeDT *n1, struct NodeTreeDT *n2);
			struct ListNodeTreeDT *allocate_listnode(void);
			void left_update(void);
			void right_update(void);
			void update(void);
			int rec_update(struct NodeTreeDT *n);
			void free_subtree(struct NodeTreeDT *n, int last);
			void free_node(struct NodeTreeDT *n);
			long finalize_subtree(struct NodeTreeDT *n, long min, long depth);
			void free_listnode(struct ListNodeTreeDT *ln);
			void finalize_sort(void);
			void free_sortlist(void);
			void allocate_sortlist(void);
			
			// From pattern.c
			void add_pattern(struct NodeTreeDT *n) {};

			// From score.c
			void calculate_scores(char *status, double *table);
			void rec_calc_scores(struct NodeTreeDT *, char *, double *);
			double new_score(long, long);
			double new_score2(long, long);
			//void write_subscores(FILE *f, long m);
			
			// From permu_i.c
			void alloc_permu(void);
			void free_permu(void);
			void permu_phase1(int side, long m, double *bestDistribution=NULL);
			void permu_best_score(int side, long m, double observed_score);
// 			void permu_best_score2(int side, long m, TreeOfHaplotypeTUCounts *, double observed_score);
			void permu_best_score2(int side, long m, HaplotypeTUCountsTree *, double observed_score);
			void permu_phase2(long m, double **d=NULL);
			double total_p(double p);
 			
			double calc_permu_score2(double *, int maxK, int &best_k);
			double calc_permu_score3(double *, long);
			double pvalue(long, long);	
			
			/**
			* Create a tree object from original algorithm nodes.
			* @param n Root of the original tree
			* @return new created tree
			*/
			TreeOfHaplotypeTUCounts * translateTree(struct NodeTreeDT *n);
			HaplotypeTUCountsTree * translateTree_b(struct NodeTreeDT *n);
			
			/**
			* resets the frequencies in the tree to the original values (it changes after permutations)
			* @param t Tree of haplotypes
			*/
			void resetFrequencies(TreeOfHaplotypeTUCounts *t);
			void resetFrequencies(TreeOfHaplotypeTUCounts *t, HaplotypeTUCountsVector * vector);	
			
			void resetFrequencies(HaplotypeTUCountsTree *t);
			
			/**
			* Calculate score using a Tree of haplotypes
			* @param t Tree of haplotypes
			* @param table List of values calculated.
			* @param udpateBestNodes Updates the best nodes in t
			*/
			void calculate_scores(TreeOfHaplotypeTUCounts *t, double *table, bool updateBestNodes=false);
			void calculate_scores(HaplotypeTUCountsTree *t, double *table, bool updateBestNodes=false);

			/**
			* Recursively calculate score using a Tree of haplotypes (starting at the node given)
			* @param t Tree of haplotypes
			* @param node node in the tree
			* @param table List of values calculated.
			* @param updateBestNodes If true, updates the best nodes in t
			*/
			void recTreeScore(TreeOfHaplotypeTUCounts *t, PNode node, double *table, bool updateBestNodes);
			void recTreeScore(HaplotypeTUCountsTree *t, HaplotypeTUCountsTree::iterator node, double *table, bool updateBestNodes);
			
			
	 virtual	Associationtable*  set(TUCounts* aTUCounts=NULL);
	 virtual void onePrint( ostream&, SampleTUCounts* aTuCounts, Associationtable* aTDTtable, bool whole);		
	 HaplotypeTUCountsVector * inferVector( HaplotypeTUCountsVector * v1, HaplotypeTUCountsVector * v2);
	 double calculate_best_score(TreeOfHaplotypeTUCounts *t);
	 
	 /**
	 * Create model tables (two for each marker).
	 * The trees and best subtrees for each marker must have been previously calculated and stores in vectorOfTrees1 and vectorOfTrees2.
	 */
	 void createTDTtables();
	 
	 /**
	 * Calculates the z static for table with groups and nodes given by tree 
	 * @param table Table with one haplotype per column
	 * @param tree Tree with the best nodes
	 * @return statistic for the best groups of tree with frequencies of table
	 */
	 double getStatistic(TDTtable<T> *table, HaplotypeTUCountsTree * tree);
// 	 double getStatistic(TDTtable *table, TreeOfHaplotypeTUCounts * tree);
	 
	 void updateModel(TreeDTModel *m, SampleTUCounts * t);
	 
	 void createModel0();
	 double getPValFromModel(TreeDTModel *model);
	 double getPValFromModel2(TreeDTModel *model);
	 double getPValFromModel3(TreeDTModel *model);
	 double combinePvalues(double p_value_left, double p_value_right);
	 double permu_phase2_fromModel(TreeDTModel* model);
	 double permu_phase1_fromModel(TreeDTModel* model);
	 void allocate_distribution(double ** &d);
	 void free_distribution(double ** &d);
	 void level1Distribution(int side, long m, double observed_score, double *distribution);
	 double calculatePvalFromBinomial(Associationtable *table, HaplotypeTUCountsTree * tree);

	 double getStatistic(Associationtable *tableLeft, HaplotypeTUCountsTree * leftTree, Associationtable *tableRight, HaplotypeTUCountsTree * rightTree);
			

	};



/*
{

}
*/

};

#endif
