#ifndef Dsvd_CPP
#define Dsvd_CPP 1


// double precision vector, matrix arithmetic
// Dave Curtis, 1998
namespace BIOS {

int dsvd::OK(void)
// Check that memory is allocated appropriately by constructor etc.
{ return UT.size()
         && W.size()
         && W.size()==VT.size(); }

dsvd::dsvd(int hi,int wi)
 : UT(wi,hi), W(wi,wi), VT(wi,wi)
// Constructor, only allocate memory for transposed matrices for now
 { done=0; gotU=gotV=gotWinv=0; }

int dsvd::setWinv(double lim)
// Set up inverse matrix of weights, zeroing inverses of small weights
// (as in Numerical Recipes in C, Press et al)
// Helps deal with ill-conditioned matrices, it says here
{
int i;
if (!done) return 0;
if (gotWinv) return 1;
if (!Winv.resize(W.wide,W.high)) return 0;
Winv=W;
double highest=0.0;
for (i=0;i<Winv.wide;++i)
  if (Winv[i][i]>highest) highest=Winv[i][i];
lim *=highest;
for (i=0;i<Winv.wide;++i)
  if (Winv[i][i]<lim) Winv[i][i]=0;
  else Winv[i][i]=1/Winv[i][i];
gotWinv=1;
return 1;
}

int dsvd::setU()
// If U is needed, get from UT (transposed U)
{
if (!done) return 0;
if (gotU) return 1;
if (!U.resize(UT.wide,UT.high)) return 0;
for (int i=0;i<U.high;++i)
  for (int j=0;j<U.wide;++j)
    U[i][j]=UT[j][i];
return 1;
}

int dsvd::setV()
// If V is needed, get from VT (transposed V)
{
if (!done) return 0;
if (gotV) return 1;
if (!V.resize(VT.wide,VT.high)) return 0; // (actually VT is square)
for (int i=0;i<V.high;++i)
  for (int j=0;j<V.wide;++j)
    V[i][j]=VT[j][i];
return 1;
}

void dsvd::zerod(double lim)
// Set to zero all weights less than lim*highest weight
 {
 double highest=0.0;
 int i;
 for (i=0;i<W.wide;++i)
  if (W[i][i]>highest) highest=W[i][i];
 lim *=highest;
 for (i=0;i<W.wide;++i)
  if (W[i][i]<lim) W[i][i]=0;
 }


dv2d dsvd::solve(dv2d const & b)
// Solve for X in AX=b, where A is original matrix before decomposition
 {
 dv2d temp(W.wide,b.wide);
 setV();
 setWinv();
 temp=V*Winv*UT*b;
 return temp;
 }

#if 0
dv2d dsvd::svbksb(dv2d& b)
// Solve for X in AX=b where A is original matrix before decomposition
// Adapted from Numerical recipes in C, Press et al
   {
   dv2d x(W.wide,1);  // as high as W is wide
   int jj,j,i;
   double s;
   setU(); setV(); // could rewrite this later so won't need them
   int n=U.wide,m=U.high;
   doublevec tmp(n+1);
   for (j=0;j<n;j++) {
      s=0.0;
      if (W[j][j]) {
         for (i=0;i<m;i++) s += U[i][j]*b[i][0];
         s /= W[j][j];
      }
      tmp[j]=s;
   }
   for (j=0;j<n;j++) {
      s=0.0;
      for (jj=0;jj<n;jj++) s += V[j][jj]*tmp[jj];
      x[j][0]=s;
   }
return x;
}


#endif


#ifndef MAX
#define MAX(a,b)  ( (a) > (b) ? (a) : (b) )
#define MIN(a,b)  ( (a) < (b) ? (a) : (b) )
#endif

int dsvd::dcmp(dv2d &A)
{
// C++ wrapper for dsvdc single value decomposition routine, which
// is modified from clinpack csvdc routine - Dave Curtis 1998-2002

    int info,i,j,high,wide;
    high=A.get_height();
    wide=A.get_width();
	double ctemp, **a, **x, **ut, **vt, *s, *e, *work;
	doublevec wwork(high), ss(MIN(wide,high+1)),ee(wide);
	dv2d aa(wide,MAX(high,wide)),tempUT(0,0);
//	aa is A in column-row format, zero-padded if necessary
    for (i=0;i<MAX(high,wide);++i)
      for (j=0;j<wide;++j)
        { aa[j][i]=i<high?A[i][j]:0; }
	a=aa.vec();
	work=wwork.vec(); s=ss.vec(); e=ee.vec();
	if (wide<=high)
	  ut=UT.vec();
	else
	  {
	  tempUT.resize(wide,wide);
	  ut=tempUT.vec();
	  // extra rows, have to copy into UT later
	  }
	vt=VT.vec();

dsvdc( a, high, wide, s, e, ut, vt, work, 21, &info );

if (info!=0) return 0; // failed
if (wide>high)
  for (i=0;i<high;++i)
    for (j=0;j<wide;++j)
        UT[j][i]=j<high?tempUT[j][i]:0;
/* set columns >= high to zero so as not to confuse testing routine
   they should not be used as undefined and all W[i][i], i>=high seem
   to be zero, though below in fact only does this for i>=high+1
*/
for (i=0;i<wide;++i)
    for (j=0;j<wide;++j)
        W[i][j]=(j==i && i<MIN(wide,high+1))?s[i]:0;
done=1;
return 1;
}


}


#endif
