/* File: genotype.h */

#include <string>
#include "Exceptions.h"



#ifndef __genotype_h__
#define __genotype_h__



namespace SNP {

/************************/
/* SNP'S GENOTYPE DEFINITION */
/************************/


/**
        @memo Genotype for SNPs

	@doc
        Definition:
        A unordered pair of SNPs values for an individual and a genetic position.
        One has been transmitted from the father, one for the mother. Who transmits
        each one of them does not matter.

        Memory space: O(1). 

        @author Maria M. Abad
	@version 1.0
*/

class genotype {

  protected:
    /** @name Implementation of class genotype
        @memo Private part.
    */


      /**
      @memo Declaration of type gender
      @doc It can have one of {1,2} values. 1 for male, 2 for female.
      */

      /**
       @memo Number of SNPs
       @doc  Used to declare an array of this length
    */

      unsigned int TotalSNPs;


	  
   /** @name Number of individuals (sample size)
        @doc Is the total number of individuals in the sample.
    */
   
    int Size;


      /**
       @memo Definition of type allele
       @doc
        Used to declare alleles in SNPs. They can have one of the values 0,1,2,3,4,
        representing missing, adenine, cytosine, guanine, thymine respectively.
    */


        // adenine     A=1;
        // cytosine     C=2;
        // guanine    G=3;
        // thymine   T=4;

     typedef enum allele {N=0, A, C, G, T};

	
	 /** @name Pointer to the list of the wild-type allele for every SNP
        @memo The wild-type will be the more frequent allele in the dataset
    */


        
	 allele* MajorAllele;

	 

	 /** @name Pointer to the list of the mutant allele for every SNP
        @memo The mutant will be the less frequent allele in the dataset
    */


        
	 allele* MinorAllele;

     

   typedef struct Genotype {

      /**
      @memo A pointer to the list of left alleles
      @doc  Each left allele contains a value {0,1,2,3,4}
      */
      allele *Left;

      /**
      @memo A pointer to the list of right alleles
      @doc Each right allele contains a value {0,1,2,3,4}
      */
      allele *Right;

	   /**
      @memo next
      @doc It contains a pointer to the phenotype of the following individual
      */

      Genotype *Next;

	  
	  /**
      @memo next
      @doc It contains a pointer to the phenotype of the previous individual
      */
      Genotype *Previous;

   };  // end structure genotype


/**
   @memo Pointer to the array of snps. It's NULL if snps has not been assigned yet.
*/

    Genotype * TheFirstGenotype;


/*********************************************************************/
/***     ASSOCIATED FUNCTIONS     ***/
/*********************************************************************/

/* Head */
      /**
         @memo Destroy the vector of SNP genotypes for an individual
	 @doc
           release memory for this vector.

      */
      void destroy (Genotype * IndGenotype);

	  /**
         @memo Copy the vector of genotypes
         @param target: pointer where will be copied
         @param origen: pointer to the vector of genotypes to copy
	*/
          void copy(Genotype * target, const Genotype * origen);

		  void ReadGenotypes (Genotype * TheFirstGenotype, ifstream * Origen);



      /* PUBLIC FUNCTIONS (INTERFACE) */

      public:



      /** @name Operations on genotype 
        @memo Operations on a genotype 
    */

      /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */
		  genotype(const unsigned int TotalSNPs);

      /**
         @memo Copy constructor
         @param destino: genotype where will be copy
         @param origen: genotype to copy
         @doc
           Make a copy of genotype
           Time complexity in time O(1).
        */
		  genotype (const genotype & origen);


/**
         @memo From file constructor
         @param destino: genotype where will be copy
         @param origen: file with genotypes to copy
         @doc
           Make a copy of genotype
           Time complexity in time O(1).
        */
		  genotype (const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize);


      /**
	 @memo Destructor
	 @doc
           Deallocate memory used by snps.
           Time complexity O(1).

      */
      ~genotype ();

      /**
         @memo Assignation
         @param g genotype to copy.
         @return Reference to the receptor genotype.
	 @doc
           Copy the genotype in the receptor genotype.
           Time complexity O(1).

      */
      genotype& operator=(const genotype& g);


	    /////////////////////////////////////////////
      /**
         @memo Constructor from input buffer
         @param file: file position in which is the individual who will be copy
         @param origen: individual to read
         @doc
           Read an individual
           Time complexity in time O(1).
        */
      genotype (const genotype & origen, istream* file);

	   /**
         @memo Obtain the k genotype in the sample.
         @param The number k
         @return return a pointer to the k genotype in the sample
         Time complexity O(TotalSNPs)

      */

	  Genotype* GetGenotype(const unsigned int k);


      /**
         @memo Check if the genotype is non missing.
         @param g: pointer to the genotype to check.
         @return bool
         @doc Return a false if one or both alleles in the genotype are missing,
         return a true if both are non missing. Complexity O(1).

      */
        bool IsANonMissingSNP (const Genotype* IndGenotype, const unsigned int position);   


      /**
         @memo Check if the genotype is heterozygous
         @param g: pointer to the genotype to check.
         @return bool 
         @doc Return a false if the genotype snp is not heterozygous
         (it can be missing, homozygous wild type or homozygous mutant).
         return a true if the genotype snp is heterozygous.
         Complexity O(1).

      */
        bool IsHeterozygous (const Genotype* IndGenotype, const unsigned int position);             
      
		/**
         @memo Check if the genotype is homozygous for wild-type allele
         @param position: pointer to the genotype to check,
         allele to check homozygosity.
         @return bool 
         @doc Return a false if the genotype snp is not homozygous wild-type
         (it can be missing, homozygous mutant or heterozygous).
         return a true if the genotype snp is homozygous wild-type.
         Complexity O(1).

      */
        bool IsHomozygous1 (const Genotype* IndGenotype, const unsigned int position);             

/**
         @memo Check if the genotype is homozygous for mutant allele
         @param position: pointer to the genotype to check,
         allele to check homozygosity.
         @return bool 
         @doc Return a false if the genotype snp is not homozygous wild-type
         (it can be missing, homozygous mutant or heterozygous).
         return a true if the genotype snp is homozygous wild-type.
         Complexity O(1).

      */
        bool IsHomozygous2 (const Genotype* IndGenotype, const unsigned int position);             


	/**
         @memo Check if a pair of genotypes (used for parents) is homozygous for wild-type allele
         @param position: pointers to the genotypes to check,
         alleles to check homozygosity.
         @return bool 
         @doc Return a false if at least one of the genotype snp is not homozygous wild-type
         (it can be missing, homozygous mutant or heterozygous).
         return a true if both the genotypes snp are homozygous wild-type.
         Complexity O(1).
      */
        bool IsHomozygous1Homozygous1 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

		
	/**
         @memo Check if a pair of genotypes (used for parents) is homozygous for mutant allele
         @param position: pointers to the genotypes to check,
         alleles to check homozygosity.
         @return bool 
         @doc Return a false if at least one of the genotype snp is not homozygous mutant
         (it can be missing, homozygous wild-type or heterozygous).
         return a true if both the genotypes snp are homozygous mutant.
         Complexity O(1).
      */
        bool IsHomozygous2Homozygous2 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

				
	/**
         @memo Check if a pair of genotypes (used for parents) is homozygous for different types
         @param position: pointers to the genotypes to check,
         alleles to check homozygosity1 and homozygosity2.
         @return bool 
         @doc Return a false they are not homozygous1 and homozygous2 
         return a true if one genotype snp is homozygous wild-type and the other mutant.
         Complexity O(1).
      */
        bool IsHomozygous1Homozygous2 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

	/**
         @memo Check if a pair of genotypes (used for parents) are homozygous1 and heterozygous
         @param position: pointers to the genotypes to check,
         alleles to check homozygosity1 and heterozygosity.
         @return bool 
         @doc Return a false they are not homozygous1 and heterozygous 
         return a true if one genotype snp is homozygous wild-type and the other heterozygous.
         Complexity O(1).
      */
        bool IsHomozygous1Heterozygous (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

		/**
         @memo Check if a pair of genotypes (used for parents) are homozygous2 and heterozygous
         @param position: pointers to the genotypes to check,
         alleles to check homozygosity2 and heterozygosity.
         @return bool 
         @doc Return a false they are not homozygous2 and heterozygous 
         return a true if one genotype snp is homozygous mutant and the other heterozygous.
         Complexity O(1).
      */
        bool IsHomozygous2Heterozygous (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

			/**
         @memo Check if a pair of genotypes (used for parents) are heterozygous
         @param position: pointers to the genotypes to check,
         alleles to check heterozygosity.
         @return bool 
         @doc Return a false they are not heterozygous 
         return a true if both are heterozygous.
         Complexity O(1).
      */
        bool IsHeterozygousHeterozygous (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

	/**
         @memo Check if a pair of genotypes (used for parents) are both missing
         @param position: pointers to the genotypes to check,
         alleles to check double missingness.
         @return bool 
         @doc Return a false they if at least one is not missing 
         return a true if both genotypes at SNP are missing.
         Complexity O(1).
      */
        bool IsMissingMissing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

	/**
         @memo Check if a pair of genotypes (used for parents) are one homozygous wild-type the other missing
         @param position: pointers to the genotypes to check,
         alleles to check one missing, the other homozygous wild-type.
         @return bool 
         @doc Return a true if one genotype at SNP is missing and the other homozyogus wild-type.
		 Otherwise return false
         Complexity O(1).
      */
        bool IsHomozygous1Missing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      


		/**
         @memo Check if a pair of genotypes (used for parents) are one homozygous mutant the other missing
         @param position: pointers to the genotypes to check,
         alleles to check one missing, the other homozygous mutant.
         @return bool 
         @doc Return a true if one genotype at SNP is missing and the other homozyogus mutant.
		 Otherwise return false
         Complexity O(1).
      */
        bool IsHomozygous2Missing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

	
		/**
         @memo Check if a pair of genotypes (used for parents) are one heterozygous and the other missing
         @param position: pointers to the genotypes to check,
         alleles to check one missing, the other heterozygous.
         @return bool 
         @doc Return a true if one genotype at SNP is missing and the other heterozyogus.
		 Otherwise return false
         Complexity O(1).
      */
        bool IsHeterozygousMissing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position);      

      /**
         @memo Is equal at position p
         @param g: genotype to compare with and position of the SNP.
	 @return
           Return true if the genotype at this position is the same, false otherwise.
         @doc Time complexity O(1).

      */
      bool IsEqual(const Genotype* IndGenotype, const Genotype*  g, const unsigned int position);


      /**
         @memo Is equal
         @param g: genotype to compare with.
	 @return
           Return true if all the genotype is the same, false otherwise.
         @doc Time complexity O(1).

      */
      bool operator==(const genotype & g);
      /**
         @memo Is different
         @param g, position: genotype to compare with.
	 @return
           Return true if the SNP is different, false otherwise.
         @doc Time complexity O(1).

      */
      bool operator!=(const genotype & g);

        /**
         @memo Obtain the sample Size.
         @param g: the sample whose size is being obtained.
         @return the size of a sample
         @doc Return the number of phenotypes in the sample. This value
         is in the variable Size.
         Time complexity O(1)

      */
        unsigned int GetSize ();          
		

		
	        /**
         @memo Obtain the number of SNPs.
         @return the number of SNPs
         @doc Return the number of SNPs. This value
         is in the variable TotalSNPs.
         Time complexity O(1)

      */
        unsigned int GetTotalSNPs ();          



		  /**
         @memo Write the genotypes in a new file.
         @param OutputFile: file where Genotypes will be written.
         @doc Write genotypes that can have been modified
         Time complexity O(Size*TotalSNPs)

      */

	  void WriteGenotypes (char* filename);

	  /**
         @memo Write the genotypes in a new file.
         @param OutputFile: file where Genotypes will be written.
         @doc Write genotypes that can have been modified
         Time complexity O(Size*TotalSNPs)

      */
	  void PrintGenotype (unsigned int Position);


};  // End of class genotype



/**********************************/
/* DEFINITIONS OF THE FUNCTIONS */
/**********************************/

//////////// privates /////////////



void genotype::destroy(Genotype * IndGenotype)
{
 assert (IndGenotype != NULL);
 assert (IndGenotype->Left != NULL);
 assert (IndGenotype->Right != NULL);
 delete IndGenotype->Left;
 delete IndGenotype->Right;
 destroy (IndGenotype->Next);
 delete IndGenotype;
};

/*___________________________________________________________ */

void genotype::copy(Genotype * Target, const Genotype * Origen)
{

 for (int i=0; i<TotalSNPs;i++)
 {
  *((Target->Left)+i)=*((Origen->Left)+i);
  *((Target->Right)+i)=*((Origen->Right)+i);
 }
//PrintGenotype(Size);
 		 
 if (Origen->Next!=NULL)
 {
  if ((Target->Next=new Genotype)==NULL)
   throw NoMemory();
  if ((Target->Next->Left=new allele[TotalSNPs])==NULL)
   throw NoMemory();
  if ((Target->Next->Right=new allele[TotalSNPs])==NULL)
   throw NoMemory();
  Target->Next->Previous=Target;
  copy(Target->Next,Origen->Next);     
 }  
 else Target->Next=NULL;
Size++;
}


/*___________________________________________________________ */

void genotype::ReadGenotypes (genotype::Genotype * target, ifstream * origen)
{
	char *genotypebuf, *cad, cadena[2];
    if ((genotypebuf=new char[TotalSNPs*4+40])==NULL)
     throw NoMemory();

	origen-> getline (genotypebuf, TotalSNPs*4+40, '\n');

	cad = strtok (genotypebuf," \t");

	for (int phen=1;phen<7;phen++)
     cad = strtok (NULL," \t");

    for (int cont=0;cont<TotalSNPs;cont++)
    {
	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	*((target->Left)+cont)=(allele) atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	*((target->Right)+cont)=(allele) atoi(cadena);
    }
	
	assert (genotypebuf!=NULL);
	delete genotypebuf;
	
    if (origen->peek()!=EOF)
	{
	if ((target->Next=new Genotype)==NULL)
      throw NoMemory();
	if ((target->Next->Left=new allele[TotalSNPs])==NULL)
      throw NoMemory();
    if ((target->Next->Right=new allele[TotalSNPs])==NULL)
      throw NoMemory();
    target->Next->Previous=target;
    ReadGenotypes (target->Next, origen);
	}
	else target->Next=NULL;

    Size++;
}


///////////////////
//// public ////////
///////////////////


genotype::genotype(const unsigned int TotalSNPs)
{
 // this->TheFirstGenotype=NULL;
  this->TotalSNPs=TotalSNPs;

   if ((TheFirstGenotype=new Genotype[TotalSNPs])==NULL)
                throw NoMemory();
}

/*____________________________________________________________ */

genotype::genotype (const genotype& origen)
{

  TotalSNPs=origen.TotalSNPs;
  
  Size=0;

  if (&origen==NULL)
    TheFirstGenotype=NULL;
else 
{
 if ((TheFirstGenotype=new Genotype)==NULL)
   throw NoMemory();
 if ((TheFirstGenotype->Left=new allele[TotalSNPs])==NULL)
   throw NoMemory();
if ((TheFirstGenotype->Right=new allele[TotalSNPs])==NULL)
   throw NoMemory();

try 
{
 copy (TheFirstGenotype, origen.TheFirstGenotype);
}
catch (NoMemory wm) 
{
 wm.PrintMessage();
}
}
}



/*____________________________________________________________ */

genotype::genotype (const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize)
{

cout << "\nReading genotypes ...";
	
	TotalSNPs=InputTotalSNPs;

ifstream InputFile; 
Size=0;

try
{
 InputFile.open (filename, ifstream::in);
 if (InputFile.peek()==EOF)
  throw EmptyFile();
}
catch (ErrorFile NoFile) {
	NoFile.PrintMessage();}
catch (EmptyFile EFile) {
        EFile.PrintMessage();
   }

try {
if ((this->TheFirstGenotype=new Genotype)==NULL)
      throw NoMemory();
if ((this->TheFirstGenotype->Left=new allele[TotalSNPs])==NULL)
      throw NoMemory();
if ((this->TheFirstGenotype->Right=new allele[TotalSNPs])==NULL)
      throw NoMemory();
  genotype::ReadGenotypes (this->TheFirstGenotype, &InputFile);
  }
catch (NoMemory no) {
  no.PrintMessage();
  }

InputFile.close();


if (InputSize != Size)
{
 cerr << "There are " << Size <<" phenotypes in the file but you have especified that the number is " << InputSize;
 exit(0);
}


cout << "\nGenotype reading has finished ...";

 }



/*____________________________________________________________ */

genotype::~genotype ()
{
                destroy(TheFirstGenotype);
}

/*____________________________________________________________ */

genotype& genotype::operator=(const genotype& genotype2)
{
  if (this!=&genotype2) {
   try {
                destroy(TheFirstGenotype);
   }
        catch (NullNode n) {
                n.PrintMessage();
	}
	try {
          copy (TheFirstGenotype, genotype2.TheFirstGenotype);
	}
        catch (NoMemory NM ) {
                NM.PrintMessage();
	}
  }
  return *this;
}
/*____________________________________________________________ */

genotype::Genotype* genotype::GetGenotype(const unsigned int k)
{
  genotype::Genotype *i;

  i=TheFirstGenotype;
  try {
  for (int c=0;c<k;c++)
      i=i->Next;
  }
  catch (NullNode null) {
    null.PrintMessage();
    }

  return i;
}

/*___________________________________________________*/

bool genotype::IsANonMissingSNP (const Genotype* IndGenotype, const unsigned int position)
{
if ((*((IndGenotype->Left)+position)!=0) && (*((IndGenotype->Right)+position)!=0))
 return true;
 else return false;
}

/*___________________________________________________*/


bool genotype::IsHeterozygous (const Genotype* IndGenotype, const unsigned int position)
{
 if (IndGenotype==NULL) {
 throw NullNode();
 return 0;
}
bool Heterozygous=false;
if (IsANonMissingSNP(IndGenotype, position))
if  (((*((IndGenotype->Left)+position)==*(MajorAllele+position)) && (*((IndGenotype->Right)+position)!=*(MajorAllele+position))) ||
	 ((*((IndGenotype->Left)+position)!=*(MajorAllele+position)) && (*((IndGenotype->Right)+position)==*(MajorAllele+position))))
Heterozygous = true;
return Heterozygous;
}

/*___________________________________________________*/


bool genotype::IsHomozygous1 (const Genotype* IndGenotype, const unsigned int position)
{
if (IndGenotype==NULL) {
 throw NullNode();
 return 0;
}
bool Is=false;
if (IsANonMissingSNP(IndGenotype, position))
if (*((IndGenotype->Left)+position)==*(MajorAllele+position))
if (*((IndGenotype->Right)+position)==*(MajorAllele+position)) 
Is=true;
return Is;
}

/*___________________________________________________*/


bool genotype::IsHomozygous2 (const Genotype* IndGenotype, const unsigned int position)
{
if (IndGenotype==NULL) {
 throw NullNode();
 return 0;
}
bool Is=false;
if (IsANonMissingSNP(IndGenotype, position))
if (*((IndGenotype->Left)+position)!=*(MajorAllele+position))
if (*((IndGenotype->Right)+position)!=*(MajorAllele+position)) 
Is=true;
return Is;
};

/*___________________________________________________*/

bool genotype::IsHomozygous1Homozygous1 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if ((IsHomozygous1(IndGenotype1, position)) && (IsHomozygous1(IndGenotype2, position)))
		return true;
	else return false;
};

/*___________________________________________________*/

bool genotype::IsHomozygous2Homozygous2 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if ((IsHomozygous2(IndGenotype1, position)) && (IsHomozygous2(IndGenotype2, position)))
		return true;
	else return false;
};

/*___________________________________________________*/

bool genotype::IsHomozygous1Homozygous2 (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHomozygous1(IndGenotype1, position)) && (IsHomozygous2(IndGenotype2, position))) 
		|| ((IsHomozygous2(IndGenotype1, position)) && (IsHomozygous1(IndGenotype2, position))))
		return true;
	else return false;
};

/*___________________________________________________*/

bool genotype::IsHomozygous1Heterozygous (const Genotype* IndGenotype1,const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHomozygous1(IndGenotype1, position)) && (IsHeterozygous(IndGenotype2, position))) 
		|| ((IsHeterozygous(IndGenotype1, position)) && (IsHomozygous1(IndGenotype2, position))))
		return true;
	else return false;
};

/*___________________________________________________*/

bool genotype::IsHomozygous2Heterozygous (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHomozygous2(IndGenotype1, position)) && (IsHeterozygous(IndGenotype2, position))) 
		|| ((IsHeterozygous(IndGenotype1, position)) && (IsHomozygous2(IndGenotype2, position))))
		return true;
	else return false;
};


/*___________________________________________________*/

bool genotype::IsHeterozygousHeterozygous (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if ((IsHeterozygous(IndGenotype1, position)) && (IsHeterozygous(IndGenotype2, position))) 
		return true;
	else return false;
};


/*___________________________________________________*/

bool genotype::IsMissingMissing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if ((!IsANonMissingSNP(IndGenotype1, position)) && (!IsANonMissingSNP(IndGenotype2, position))) 
		return true;
	else return false;
};


/*___________________________________________________*/

bool genotype::IsHomozygous1Missing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHomozygous1(IndGenotype1, position)) && (!IsANonMissingSNP(IndGenotype2, position))) ||
		((!IsANonMissingSNP(IndGenotype1, position)) && (IsHomozygous1(IndGenotype2, position))))
		return true;
	else return false;
};


/*___________________________________________________*/

bool genotype::IsHomozygous2Missing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHomozygous2(IndGenotype1, position)) && (!IsANonMissingSNP(IndGenotype2, position))) ||
		((!IsANonMissingSNP(IndGenotype1, position)) && (IsHomozygous2(IndGenotype2, position))))
		return true;
	else return false;
};

/*___________________________________________________*/

bool genotype::IsHeterozygousMissing (const Genotype* IndGenotype1, const Genotype* IndGenotype2, const unsigned int position)      
{
	if (((IsHeterozygous(IndGenotype1, position)) && (!IsANonMissingSNP(IndGenotype2, position))) ||
		((!IsANonMissingSNP(IndGenotype1, position)) && (IsHeterozygous(IndGenotype2, position))))
		return true;
	else return false;
};


/*___________________________________________________*/


bool genotype::IsEqual(const Genotype* IndGenotype, const Genotype* g, const unsigned int position)
{
bool Is=false;
if (*((IndGenotype->Left)+position)==*((g->Left)+position))
if (*((IndGenotype->Right)+position)==*((g->Right)+position))
Is=true;
return Is;
}

		
/*____________________________________________________________ */

unsigned int genotype::GetSize()
{
  return Size;
}
	
/*____________________________________________________________ */

unsigned int genotype::GetTotalSNPs()
{
  return TotalSNPs;
}
/*____________________________________________________________ */

void genotype::PrintGenotype (unsigned int Position)
{
Genotype* target=GetGenotype(Position);
for(int i=0;i<TotalSNPs;i++)
{
cout << *(target->Left+i) << ' ';
cout << *(target->Right+i) << ' ';
}
	}
/*____________________________________________________________ */


void genotype::WriteGenotypes (char* filename)
{
Genotype * IndGenotype;
ostream* OutputFile; 

try
{
 OutputFile=new ofstream(filename);
}
catch (ErrorFile NoFile) {
        NoFile.PrintMessage();
      }
IndGenotype=TheFirstGenotype;
for (int i=0; i<Size;i++)
{
 for (int i2=0; i2<TotalSNPs;i2++)
 {
  (*OutputFile) << *((IndGenotype->Left)+i2) << ' ';
  (*OutputFile) << *((IndGenotype->Right)+i2) << ' ';
 }
  (*OutputFile) << "\n";
  IndGenotype=IndGenotype->Next;
}

//OutputFile->close();
}

};  // End of Namespace

#endif

/* End of file: genotype.h */




