/* File: individual.h */

#include <string>


#include "Exceptions.h"

#ifndef __individual_h__
#define __individual_h__

#include "genotype.h"
#include "phenotype.h"

namespace SNP {

#define  maxint  32767//


   

//   const short unsigned int male=1;
//   const short unsigned int female=2;
//   const short unsigned int unaffected=0;
//   const short unsigned int affected=2;


/************************/
/* individual DEFINITION */
/************************/


/**
        @memo individual for SNPs

	@doc
        Definition:
        A set of phenotype's features and genotypes for an individual

        Memory space: O(TotalSNPs), which TotalSNPs being the number of SNPs in the sample
        Each individual in a sample has been genotyped for the same TotalSNPs SNPs.

        @author Maria M. Abad
	@version 1.0
*/

   class individual: public phenotype, public genotype 
   {
   protected:
    /** @name Implementation of class individual
        @memo Private part.
    */


        
//	 typefile tf;




/***************************************************************/
/***     ASSOCIATED FUNCTIONS     ***/
/*********************************************************************/

/* Head */


/////////////////////////



      /* PUBLIC FUNCTIONS (INTERFACE) */

      public:


		  
		  /**
         @memo Creates a new individual objet with the phase resolved.
         @param Origen: the origianl individual object 
         Time complexity O(TotalSNPs*Size*Size)

      */
	  individual* ResolvePhase (individual* Origen);   


	  

      /**
         @memo Write a new file with resolved phase.
         @param filename: the filename where results will be written
         Time complexity O(TotalSNPs*Size)

      */
	/*  void WriteResults (char* filename);   
    /**
	 @memo Destructor
	 @doc
           Deallocate memory used by individual.
           Time complexity O(1).

      */
	  ~individual ();


	   /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */
	  individual(const unsigned int TotalSNPs, const unsigned int Size, bool ExistPhenotype);


  /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */

	individual(const individual& origen, IndCategory ic); 

//	  individual(const phenotype& porigen, const genotype& gorigen, IndCategory ic);

 /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */

		individual(const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize, bool ExistPhenotype, IndCategory ic); 

	  
	  /**
         @memo Increase the count the total number of known haplotypes between two positions.
         @param comb: pointer to a 5-integer array: 1: 11, 2:12, 3: 21, 4: 22
         @param IndGenotype: individual whose haplotype is going to be checked
         @param FirstSNP, LastSNP: the two loci
         @param MajorAllele: pointer to a a TotalSNPs allele table with the major allele for each SNP
     */

	     void CountHaps (unsigned int* comb, genotype::Genotype* IndGenotype, unsigned int FirstSNP, unsigned int LastSNP);


 /**
         @memo Resolve the phase between two consequtive heterozygous position for an individual.
         @param comb: pointer to a 5-integer array: 1: 11, 2:12, 3: 21, 4: 22
         @param ForPhase: pointer to the genotypes of all individuals
         @param FirstSNP, LastSNP: the two loci
         @param MajorAllele: pointer to a a TotalSNPs allele table with the major allele for each SNP
	     @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
*/

	   void ObtenerBestPhase (unsigned int* comb, unsigned int FirstSNP, unsigned int LastSNP, IndCategory ic);


/**
         @memo Resolve the phase for a data set.
         @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
*/
		void ResolvePhase (IndCategory ic);




/**
         @memo Compute the more frequent allele for each SNP.
         @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
		 @ return: pointer to a table of TotalSNPs alleles with the more frequent value
*/

		 void GetMajorAllele(IndCategory ic);

/**
         @memo Compute the less frequent allele for each SNP.
         @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
		 @ return: pointer to a table of TotalSNPs alleles with the less frequent value
*/

		 void GetMinorAllele(IndCategory ic);



/**
         @memo Order all the SNPs genotypes in the sample.so when a genotype for a individual is
		 heterozygous, the one in the Left positions must be the Major allele.
         @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
         @param MajorAllele: pointer to a a TotalSNPs allele table with the major allele for each SNP
*/

		void OrderSNPs(IndCategory ic);


/**
         @memo Resolve phase for every pair of consecutive heterozygous positions for every individual in the sample
         @param ic: 0: if phase has to be resolved only for parents, 1: only for children, 2 for everybody
         @param MajorAllele: pointer to a a TotalSNPs allele table with the major allele for each SNP
*/

		unsigned int AssignPhase (unsigned int FirstHetero, unsigned int LastHetero, IndCategory ic, unsigned int OldMayorPhase);



};  // End of class individual



/**********************************/
/* DEFINITIONS OF THE FUNCTIONS */
/**********************************/


///////////////////
//// public ////////
///////////////////


/*____________________________________________________________ */

individual::individual(const unsigned int TotalSNPs, const unsigned int Size, bool ExistPhenotype): 
phenotype(Size, ExistPhenotype), genotype(TotalSNPs) 
{
};

/*____________________________________________________________ */


individual::individual(const individual& origen, IndCategory ic): phenotype(origen), genotype(origen) 
{
cout << "\nComputing major and minor alleles ...";

try {
 GetMajorAllele (ic);
 GetMinorAllele (ic);
  }
catch (NoMemory no) {
  no.PrintMessage();
  }
  cout << "\nComputation of major and minor alleles has finished";

};

/*____________________________________________________________ */

individual::individual(const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize, bool ExistPhenotype, IndCategory ic): 
phenotype(filename, InputSize, InputTotalSNPs, ExistPhenotype, ic), genotype(filename, InputTotalSNPs, InputSize) 
{

//	cout << "\nComputing major and minor alleles ...";
try {
// GetMajorAllele (ic);
// GetMinorAllele (ic);
  }
catch (NoMemory no) {
  no.PrintMessage();
  }

// cout << "\nComputation of major and minor alleles has finished";

};


/*_____________________________________________________________*/


void individual::CountHaps (unsigned int* comb, genotype::Genotype* IndGenotype, unsigned int FirstSNP, unsigned int LastSNP) 
//count known haplotypes
{
 if (genotype::IsHomozygous1(IndGenotype, FirstSNP) && IsHomozygous1(IndGenotype, LastSNP))         
  comb[0]=comb[0]+2;// 1 1 
 if (IsHomozygous1(IndGenotype, FirstSNP) && IsHomozygous2(IndGenotype, LastSNP))         
  comb[1]=comb[1]+2;// 1 2 
 if (IsHomozygous2(IndGenotype, FirstSNP) && IsHomozygous2(IndGenotype, LastSNP))         
  comb[3]=comb[3]+2;// 2 2 
 if (IsHomozygous2(IndGenotype, FirstSNP) && IsHomozygous1(IndGenotype, LastSNP))         
  comb[2]=comb[2]+2;// 2 1         
 if (IsHomozygous1(IndGenotype, FirstSNP) && IsHeterozygous(IndGenotype, LastSNP))         
 {
       comb[0]++; // 1 1 
       comb[1]++; // 1 2
 }
 if (IsHomozygous2(IndGenotype, FirstSNP) && IsHeterozygous(IndGenotype, LastSNP))         
 {
       comb[2]++; // 2 1 
       comb[3]++; // 2 2
 }
 if (IsHeterozygous(IndGenotype, FirstSNP) && IsHomozygous1(IndGenotype, LastSNP))         
 {
	 comb[0]++;// 1 1 
     comb[2]++; // 2 1
 }
 if (IsHeterozygous(IndGenotype, FirstSNP) && IsHomozygous2(IndGenotype, LastSNP))         
 {
	 comb[1]++; // 1 2 
     comb[3]++; // 2 2
 }

}  
    	 
/*_________________________________________________________________________*/

void individual::ObtenerBestPhase (unsigned int* comb, unsigned int FirstSNP, unsigned int LastSNP, 
IndCategory ic)
// this function computes the number of 11 12 21 and 22 haplotypes existing
// in the dataset for those children with the same haplotypes for
// LeftHaplotype and RightHaplotype.
{
Phenotype * IndPhenotype=TheFirstPhenotype;
Genotype * IndGenotype=TheFirstGenotype;

for (int i=0;i<Size;i++)
{
 if ((ic==2) || ((ic==1) && (IsAChild (IndPhenotype))) || ((ic==0) & (!IsAChild (IndPhenotype))))
  if (IsANonMissingSNP (IndGenotype, FirstSNP) && IsANonMissingSNP (IndGenotype, LastSNP)) 
   CountHaps(comb, IndGenotype, FirstSNP, LastSNP);     
IndPhenotype=IndPhenotype->Next;
IndGenotype=IndGenotype->Next;
}
}

/*__________________________________________________________*/

unsigned int individual::AssignPhase (unsigned int FirstHetero, unsigned int LastHetero, IndCategory ic, unsigned int OldMayorPhase)
{
                                 
unsigned int Frequencies[4], total, MajorPhase;
double numerator;

for (int i=0;i<4;i++)
 Frequencies[i]=0;     
    
ObtenerBestPhase (&Frequencies[0], FirstHetero, LastHetero, ic);
  
total=Frequencies[0]+Frequencies[1]+Frequencies[2]+Frequencies[3];
     
Frequencies[0]++; // A
Frequencies[1]++; // a
Frequencies[2]++; // B
Frequencies[3]++; // b

numerator=((Frequencies[0]/total)*(Frequencies[3]/total))-((Frequencies[1]/total)*(Frequencies[2]/total));
  
if (numerator > 0)
  MajorPhase=1;
else 
  MajorPhase=2;
 
return MajorPhase;
} 
 /*____________________________________________________________ */

void individual::ResolvePhase (IndCategory ic)
{
unsigned int LastResolved, OldMajorPhase, MajorPhase;
cout << "\nReconstructing haplotypes...";
this->OrderSNPs(ic);
Phenotype* IndPhenotype=TheFirstPhenotype;
Genotype* IndGenotype=TheFirstGenotype;

for (int i=0;i<Size;i++)
 if ((ic==2) || ((ic==1) && (IsAChild (IndPhenotype))) || ((ic==0) & (!IsAChild (IndPhenotype))))
 {
  LastResolved=0; 
 for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 {
  if (IsHeterozygous(IndGenotype, SNP))
  {
   if (LastResolved==0) //    UnknownPreviousPhase
    LastResolved=SNP;
   else  // if not the first heterozygote SNP 
   {
    if (((*((IndGenotype->Left)+LastResolved)==*(MajorAllele+LastResolved)) && 
		(*((IndGenotype->Left)+SNP)==*(MajorAllele+SNP))) || 
		((*((IndGenotype->Left)+LastResolved)==*(MinorAllele+LastResolved)) && 
		(*((IndGenotype->Left)+SNP)==*(MinorAllele+SNP))))
     OldMajorPhase=1;
    else 
	 OldMajorPhase=2;
          
    MajorPhase=AssignPhase (LastResolved, SNP, ic, OldMajorPhase);
	MajorPhase=1;

    if (*((IndGenotype->Left)+LastResolved)==*(MajorAllele+LastResolved))
	{
     if (MajorPhase==1)
	 {
      (*((IndGenotype->Left)+SNP)=*(MajorAllele+SNP));
	  (*((IndGenotype->Right)+SNP)=*(MinorAllele+SNP));
	 }
	 if (MajorPhase==2)
	 {
      (*((IndGenotype->Left)+SNP)=*(MinorAllele+SNP));
	  (*((IndGenotype->Right)+SNP)=*(MajorAllele+SNP));
	 }
	}
     else // left last hetero SNP was phased as a 2
     {
     if (MajorPhase==1)
	 {
      (*((IndGenotype->Left)+SNP)=*(MinorAllele+SNP));
	  (*((IndGenotype->Right)+SNP)=*(MajorAllele+SNP));
	 }
	 if (MajorPhase==2)
	 {
      (*((IndGenotype->Left)+SNP)=*(MajorAllele+SNP));
	  (*((IndGenotype->Right)+SNP)=*(MinorAllele+SNP));
	 }
	}
   if (MajorPhase!=0) LastResolved=SNP;    
   } // end not the first hetero
  } // end is hetero
 } //end for each SNP
 IndGenotype=IndGenotype->Next;
 IndPhenotype=IndPhenotype->Next;
} // end for each individual
cout << "\nReconstruction has finished";
}


/*____________________________________________________________ */
/*
void individual::WriteResults (char* filename)
 {
  ostream* OutputFile; 
  Phenotype *IndPhenotype=TheFirstPhenotype; 
  Genotype *IndGenotype=TheFirstGenotype;
  try
{
 OutputFile=new ofstream(filename);
}
catch (ErrorFile NoFile) {
        NoFile.PrintMessage();
      }
  for (int i=0; i<Size;i++)
  {
    if (tf==TDT)
	{
	(*OutputFile) << ind->Pedigree << ' ';
	(*OutputFile) << ind->Code << ' ';
	(*OutputFile) << ind->Father << ' ';
	(*OutputFile) << ind->Mother << ' ';
	(*OutputFile) << ind->Gender << ' ';
	(*OutputFile) << ind->Affectation << ' ';
	(*OutputFile) << ind->Code2 << ' ';
	(*OutputFile) << ind->Pedigree << ' ';
	(*OutputFile) << ind->Pedigree << ' ';
	}
	IndGenotype=IndGenotype->Next;
	genotypeInd->WriteGenotype(OutputFile);
    ind=ind->next;
 }
/* _____________________________________________________*/

void individual::GetMajorAllele(IndCategory ic)
{
Phenotype * IndPhenotype;
Genotype * IndGenotype;
unsigned int TotalUsed;
unsigned int Basis[5], max, MaxVal;
if (MajorAllele==NULL)
if ((MajorAllele=new allele[genotype::TotalSNPs])==NULL)
      throw NoMemory();


for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
for (int c=0;c<5;c++)
 Basis[c]=0;
IndPhenotype=TheFirstPhenotype;
IndGenotype=TheFirstGenotype;

for (int i=0;i<Size;i++)
{
 if ((ic==2) || ((ic==1) && (IsAChild (IndPhenotype))) || ((ic==0) & (!IsAChild (IndPhenotype))))
	{
     if (*((IndGenotype->Left)+SNP)!=0)
	  Basis[(int)*((IndGenotype->Left)+SNP)]++; 
	 if (*((IndGenotype->Right)+SNP)!=0)
	  Basis[(int)*((IndGenotype->Right)+SNP)]++; 
	}
  IndPhenotype=IndPhenotype->Next;
  IndGenotype=IndGenotype->Next;
}
TotalUsed=0; 
max=0;
MaxVal=0;
for (int c2=1;c2<=4;c2++)
{
if (Basis[c2]!=0) TotalUsed++;      
if (TotalUsed > 2) 
 throw MultiAllelic(); 

if (Basis[c2]>max)
{
 max=Basis[c2];
 MaxVal=c2;
}
}  
MajorAllele[SNP]=(allele)MaxVal;   
}

} 


/* _____________________________________________________*/


void individual::GetMinorAllele(IndCategory ic)
{
Phenotype * IndPhenotype;
Genotype * IndGenotype;
unsigned int TotalUsed;
unsigned int Basis[5], min, MinVal;
if (MinorAllele==NULL)
if ((MinorAllele=new allele[genotype::TotalSNPs])==NULL)
      throw NoMemory();
for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
for (int c=0;c<5;c++)
 Basis[c]=0;
IndPhenotype=TheFirstPhenotype;
IndGenotype=TheFirstGenotype;
for (int i=0;i<Size;i++)
{
 if ((ic==2) || ((ic==1) && (IsAChild (IndPhenotype))) || ((ic==0) & (!IsAChild (IndPhenotype))))
	{
     if (*((IndGenotype->Left)+SNP)!=0)
	  Basis[(int)*((IndGenotype->Left)+SNP)]++; 
	 if (*((IndGenotype->Right)+SNP)!=0)
	  Basis[(int)*((IndGenotype->Right)+SNP)]++; 
	}
  IndPhenotype=IndPhenotype->Next;
  IndGenotype=IndGenotype->Next;
}
TotalUsed=0; 
min=maxint;
MinVal=0;
for (int c2=1;c2<=4;c2++)
{
 if (Basis[c2]!=0) TotalUsed++;      

if (TotalUsed > 2) 
 throw MultiAllelic(); 

if (Basis[c2]<min)
{
 min=Basis[c2];
 MinVal=c2;
}
}  
MinorAllele[SNP]=(allele)MinVal;   
}
} 

/*____________________________________________________________ */


void individual::OrderSNPs(IndCategory ic)
{
Phenotype  *IndPhenotype=TheFirstPhenotype;
Genotype *IndGenotype=TheFirstGenotype;
allele SwitchAllele;
for (int i=0;i<Size;i++)
{
 if ((ic==2) || ((ic==1) && (IsAChild (IndPhenotype))) || ((ic==0) & (!IsAChild (IndPhenotype))))
	 for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
    if (IsHeterozygous(IndGenotype, SNP))
     if (*((IndGenotype->Left)+SNP)>*((IndGenotype->Right)+SNP))
	 {        
      SwitchAllele=*((IndGenotype->Left)+SNP);
      *((IndGenotype->Left)+SNP)=*((IndGenotype->Right)+SNP);
	  *((IndGenotype->Right)+SNP)=SwitchAllele;
     }
  IndPhenotype=IndPhenotype->Next;
  IndGenotype=IndGenotype->Next;
}
}      
            
/*____________________________________________________________ */


individual::~individual ()
{
	phenotype::destroy(TheFirstPhenotype);
	genotype::destroy(TheFirstGenotype);
	assert (MajorAllele != NULL);
	delete MajorAllele;
	assert (MinorAllele != NULL);
	delete MinorAllele;


}



};  // Fin del Namespace

#endif

/* Fin Fichero: individual.h */
