/* File: phenotype.h */

#include <string>


#include "Exceptions.h"

#ifndef __phenotype_h__
#define __phenotype_h__

namespace SNP {

  /**
      @memo Declaration of type of individual
      @doc It can have one of {0,1} values. 0 for parent, 1 for children, 2 for everybody.
      */


     typedef enum IndCategory {parent=0, offspring=1, everybody=2};

	 /**
      @memo typefile
      @doc It represents the type of file, phase or TDT
      */
   //  typedef enum typefile {phase=0, TDT=1};

//   const short unsigned int male=1;
//   const short unsigned int female=2;
//   const short unsigned int unaffected=0;
//   const short unsigned int affected=2;


/************************/
/* phenotype DEFINITION */
/************************/


/**
        @memo phenotype for SNPs

	@doc
        Definition:
        A set of phenotype's features for each individual

        Memory space: O(SizeP), which SizeP being the number of individuals in the sample

    @author Maria M. Abad
	@version 1.0
*/

class phenotype {

  protected:
    /** @name Implementation of class phenotype
        @memo Private part.
    */


   /** @name Number of individuals (sample size)
        @doc Is the total number of individuals in the sample.
    */
   
    int SizeP;

    /**
       @memo Number of SNPs
       @doc  Used to declare an array of this length
    */

      unsigned int TotalSNPs;


	  
    /**
       @memo if there is phenotype information in the sample
       @doc  boolean, 1: yes, 0, no
    */

      bool ExistPhenotype;


      /**
      @memo Declaration of type gender
      @doc It can have one of {1,2} values. 1 for male, 2 for female.
      */

     
     typedef enum gender {unknown=0, male=1, female=2};



      /**
      @memo Declaration of type affectation
      @doc It can have one of {0,2} values. 0 for unaffected, 2 for affected.
      */


     typedef enum affectation {unaffected=1, affected=2};


   struct Phenotype {

      /**
      @memo Pedigree
      @doc It contains the family code
      */
      unsigned int Pedigree;

      /**
      @memo Code
      @doc It contains a phenotype code
      */
      unsigned int Code;

      /**
      @memo Father
      @doc It contains the father's code of this phenotype
      */
      unsigned int Father;

      /**
      @memo Mother
      @doc It contains the mother's code of this phenotype
      */
      unsigned int Mother;


      /**
      @memo Gender
      @doc It contains the gender of this phenotype
      */
      gender Gender;

      /**
      @memo Affectation
      @doc It contains the affectation of this phenotype
      */
      affectation Affectation;

      /**
      @memo Code2
      @doc It contains a second code of this phenotype, sometimes supplied
      */
      unsigned int Code2;
	  
	  /**
      @memo Next
      @doc It contains a pointer to the phenotype of the following individual
      */

    Phenotype *Next;

	  
	  /**
      @memo Next
      @doc It contains a pointer to the phenotype of the Previous individual
      */
    Phenotype *Previous;

   };  // end structure Phenotype

  Phenotype *TheFirstPhenotype;

/*********************************************************************/
/***     ASSOCIATED FUNCTIONS     ***/
/*********************************************************************/

/* Head */

/////////////////////////////////////////////
void destroy(Phenotype * ind);

void copy(Phenotype * Target, const Phenotype * Origen);

void ReadPhenotype (Phenotype * Target, ifstream * Origen, IndCategory ic);

/////////////////////////



      /* PUBLIC FUNCTIONS (INTERFACE) */

      public:


      /**
         @memo Destructor
	 @doc
           Deallocate memory used by phenotype.
           Time complexity O(1).

      */
      ~phenotype ();
//////////////////////////////////////////////




      /** @name Operations on phenotype 
        @memo Operations on a phenotype 
    */

      /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */
      phenotype(const unsigned int SizeP, bool ExistPhenotype);
  /////////////////////////////////////////////
      /**
         @memo Copy constructor
         @param target: phenotype where will be copy
         @param origen: phenotype to copy
         @doc
           Make a copy of phenotype
           Time complexity in time O(1).
        */
      phenotype (const phenotype & origen);


  /////////////////////////////////////////////
      /**
         @memo Constructor from input buffer
         @param file: file position in which is the phenotype who will be copy
         @param origen: phenotype to read
         @doc
           Read an phenotype
           Time complexity in time O(1).
        */
	  phenotype (const char* filename, const unsigned int InputSizeP, const unsigned int TotalSNPs, bool ExistPhenotype, IndCategory ic);

     // phenotype (const phenotype & origen, istream* file, const typefile tf);

///////////////////////////////////////////////////
      /**
         @memo Assignation
         @param ind: phenotype to copy.
         @return Reference to the receptor phenotype.
	 @doc
           Copy the phenotype in the receptor phenotype.
           Time complexity O(1).

      */
      phenotype& operator=(const phenotype & ind);
////////////////////////////////////////////////////
   
      /**
         @memo Is equal
         @param g: phenotype to compare with.
	 @return
           Return true if the SNP is the same, false otherwise.
         @doc Time complexity O(1).

      */
      bool operator==(const phenotype & ind);
      /**
         @memo Is different
         @param g: phenotype to cmpare with.
	 @return
           Return true if the SNP is different, false otherwise.
         @doc Time complexity O(1).

      */
      bool operator!=(const phenotype & ind);

	        /**
         @memo Obtain the first phenotype in the sample.
         @return return a pointer to the first phenotype in the sample
         Time complexity O(1)

      */
	  Phenotype* GetFirst ();

      /**
         @memo Obtain the Next phenotype in the sample.
         @param The pointer to the current phenotype
         @return return a pointer to the Next phenotype in the sample
         Time complexity O(1)

      */
        Phenotype* GetNext (const Phenotype* ind);

		 /**
         @memo Check if is a child.
         @param The pointer to the current individual's phenotype
         @return return a boolean reporting if is a child
         Time complexity O(1)

      */
        bool IsAChild (const Phenotype* ind);


		/**
         @memo Obtain the position of the phenotype of the father.
         @param The pointer to the current individual's phenotype
         @return return the position for the father
         Time complexity O(1)
		**/

		unsigned int  GetFather(const phenotype::Phenotype *i);

		/**
         @memo Obtain the position of the phenotype of the mother.
         @param The pointer to the current individual's phenotype
         @return return the position for the mother
         Time complexity O(1)
		**/

		unsigned int  GetMother(const phenotype::Phenotype *i);

      /**
         @memo Obtain the k phenotype in the sample.
         @param The number k
         @return return a pointer to the k phenotype in the sample
         Time complexity O(TotalSNPs)

      */
        phenotype::Phenotype* GetPhenotype (unsigned int k);             

      /**
         @memo Obtain the sample SizeP.
         @param g: the sample whose size is being obtained.
         @return the size of a sample
         @doc Return the number of phenotypes in the sample. This value
         is in the variable SizeP.
         Time complexity O(1)

      */
        unsigned int GetSizePP ();             

  

		  /**
         @memo Write a new file with resolved phase.
         @param filename: the filename where results will be written
         Time complexity O(TotalSNPs*SizeP)

      */
	  void WriteResults (char* filename);   


		  /**
         @memo Print the phenotype for an individual.
         @param Position: Position of this individual in the sample
         Time complexity O(1)

      */
	  void PrintPhenotype (unsigned int Position);


};  // End of class phenotype



/**********************************/
/* DEFINITIONS OF THE FUNCTIONS */
/**********************************/


/////////////////////////////////////////////
void phenotype::destroy(Phenotype * ind)
{

       		assert (ind != NULL);
			destroy (ind->Next);
			delete ind;
            
}
//////////////////////////////////////////

void phenotype::copy(Phenotype * Target, const Phenotype * Origen)
{

  		 Target->Affectation=Origen->Affectation;
		 Target->Code=Origen->Code;
		 Target->Code2=Origen->Code2;
		 Target->Father=Origen->Father;
		 Target->Gender=Origen->Gender;
		 Target->Mother=Origen->Mother;
		 Target->Pedigree=Origen->Pedigree;
		 Target->Previous=Origen->Previous;
		 Target->Next=Origen->Next;

 		 
         if (Origen->Next!=NULL)
		 {
		   if ((Target->Next=new Phenotype)==NULL)
            throw NoMemory();
          Target->Next->Previous=Target;
          copy(Target->Next,Origen->Next);     
		 }  
		 else Target->Next=NULL;

		 SizeP++;

}




///////////////////
void phenotype::ReadPhenotype (Phenotype * target, ifstream*  origen, IndCategory ic)
{

	if (ExistPhenotype)
	{
	char *genotypebuf, *cad, cadena[5];
    if ((genotypebuf=new char[TotalSNPs*4+40])==NULL)
     throw NoMemory();

	origen-> getline (genotypebuf, TotalSNPs*4+40, '\n');

	cad = strtok (genotypebuf," \t");
	sscanf (cad, "%s", cadena);
	target->Pedigree=atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Code=atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Father=atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Mother=atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Gender=(gender)atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Affectation=(affectation)atoi(cadena);

	cad = strtok (NULL," \t");
	sscanf (cad, "%s", cadena);
	target->Code2=atoi(cadena);

	assert (genotypebuf!=NULL);
	delete genotypebuf;
  }
	else
	{
	target->Pedigree=0;
    target->Code=0;
    target->Father=(unsigned int) ic; // if 0, all individuals are founders, if 1 offspring
    target->Mother=1; // if 0, all individuals are founders, if 1 offspring
    target->Gender=(gender)0;
    target->Affectation=(affectation)0;
    target->Code2=0;
	}



    if (origen->peek()!=EOF)
	{
	if ((target->Next=new Phenotype)==NULL)
      throw NoMemory();

		ReadPhenotype (target->Next, origen, ic);
	}
	else target->Next=NULL;
    SizeP++;

}

///////////////////
//// public ////////
///////////////////

phenotype::phenotype(const unsigned int SizeP, bool ExistPhenotype)
{
 try
 {
 TheFirstPhenotype = new Phenotype;
 phenotype::SizeP=SizeP;
 phenotype::ExistPhenotype = ExistPhenotype;

 }
  catch (NoMemory wm) {
                wm.PrintMessage();
 }
  TheFirstPhenotype=NULL;
}

/*____________________________________________________________ */

phenotype::phenotype (const phenotype & origen)
{
  SizeP=0;
  ExistPhenotype = origen.ExistPhenotype;

  if (&origen==NULL)
   TheFirstPhenotype=NULL;
  else
  {
  try {
	   if ((TheFirstPhenotype = new Phenotype)==NULL)
	    throw NoMemory();
        copy (TheFirstPhenotype, origen.TheFirstPhenotype);
		if (SizeP!=origen.SizeP)
         throw BadSize();
  }
  catch (NoMemory no) {
                no.PrintMessage();
		}
catch (BadSize bs) {
                bs.PrintMessage();
		}
  }

}

/*____________________________________________________________ */

phenotype::phenotype (const char* filename, const unsigned int InputSizeP, const unsigned int TotalSNPs, bool ExistPhenotype, IndCategory ic)
{
cout << "\nReading phenotypes ...";

ifstream InputFile; 

SizeP=0;

phenotype::ExistPhenotype = ExistPhenotype;
phenotype::TotalSNPs=TotalSNPs;
try
{
	InputFile.open (filename, ifstream::in);
	 if (InputFile.peek()==EOF)
		throw EmptyFile();
}
catch (ErrorFile NoFile) {
	NoFile.PrintMessage();}
catch (EmptyFile EFile) {
        EFile.PrintMessage();
   }


try {
if ((this->TheFirstPhenotype=new Phenotype)==NULL)
      throw NoMemory();
phenotype::ReadPhenotype (this->TheFirstPhenotype, &InputFile, ic);
  }
catch (NoMemory no) {
  no.PrintMessage();
  }


InputFile.close();

if (InputSizeP != SizeP)
{
 cerr << "There are " << SizeP <<" phenotypes in the file but you have especified that the number is " << InputSizeP;
 exit(0);
}

cout << "\nPhenotype reading has finished ...";

}

/*____________________________________________________________ */

phenotype::~phenotype ()
{
                destroy(TheFirstPhenotype);

}



 /*____________________________________________________________ */

phenotype::Phenotype*  phenotype::GetFirst()
{

    return TheFirstPhenotype;
  }


/*____________________________________________________________ */

unsigned int  phenotype::GetFather(const phenotype::Phenotype *i)
{

  if (i==NULL) {
    throw NullNode();
	 return 0;
  }
  else {
  
	bool FoundFather = 0;
	unsigned int FatherID = i->Father, Position=0;
	Phenotype *Ind = TheFirstPhenotype;
	while ((Ind!=NULL) && (Position<SizeP) && (!FoundFather))
	{
	if	((Ind->Code == FatherID) && (Ind->Pedigree == i->Pedigree))
     FoundFather = 1;
	else
	{
    Ind=Ind->Next;
	Position=Position+1;
	}
    }
	if (FoundFather)
	 return (Position); 
	else 
	{
    throw NullNode();
	 return 0;
	}
  }
}
/*____________________________________________________________ */

unsigned int  phenotype::GetMother(const phenotype::Phenotype *i)
{

  if (i==NULL) {
    throw NullNode();
	 return 0;
  }
  else {
  
	bool FoundMother = 0;
	unsigned int MotherID = i->Mother, Position=0;
	Phenotype *Ind = TheFirstPhenotype;
	while ((Ind!=NULL) && (!FoundMother))
	{
	if	((Ind->Code == MotherID) && (Position<SizeP) && (Ind->Pedigree == i->Pedigree))
     FoundMother = 1;
	else
	{
    Ind=Ind->Next;
	Position=Position+1;
    }
    }
	if (FoundMother)
	 return (Position); 
	else 
	{
    throw NullNode();
	 return 0;
	}
  }
};

/*____________________________________________________________ */

phenotype::Phenotype* phenotype::GetNext(const phenotype::Phenotype *i)
{

  if (i==NULL) {
    throw NullNode();
	 return 0;
  }
  else {
  return i->Next;
  }
}

/*____________________________________________________________ */

bool phenotype::IsAChild (const Phenotype* ind)
{
 if ((ind->Father!=0) && (ind->Mother!=0))
  return true;
 else return false;
}

/*____________________________________________________________ */

phenotype::Phenotype* phenotype::GetPhenotype(const unsigned int k)
{
  phenotype::Phenotype *i;

  i=TheFirstPhenotype;
  try {
  for (int c=0;c<k;c++)
      i=i->Next;
  }
  catch (NullNode null) {
    null.PrintMessage();
    }

  return i;
}


/*____________________________________________________________ */


unsigned int phenotype::GetSizePP()
{
  return SizeP;
}



/*____________________________________________________________ */

void phenotype::PrintPhenotype (unsigned int Position)
{
Phenotype* target=GetPhenotype(Position);
cout << target->Pedigree << ' ';
cout << target->Code << ' ';
cout << target->Father << ' ';
cout << target->Mother << ' ';
cout << target->Gender << ' ';
cout << target->Affectation << ' ';
cout << target->Code2 << ' ';
	}


/*____________________________________________________________ */


};  // Fin del Namespace

#endif

/* Fin Fichero: phenotype.h */
