/* File: rTDT.h */

#include <string>
#include "Exceptions.h"
#include "individual.h"
#include "gamma.h"
#include <math.h>
namespace SNP {

  
/************************/
/* individual DEFINITION */
/************************/


/**
        @memo individual for SNPs

	@doc
        Definition:
        A set of phenotype's features and genotypes for an individual

        Memory space: O(TotalSNPs), which TotalSNPs being the number of SNPs in the sample
        Each individual in a sample has been genotyped for the same TotalSNPs SNPs.

        @author Maria M. Abad
	@version 1.0
*/

 class rTDT: public individual
 {

  protected:
    /** @name Implementation of class individual
        @memo Private part.
    */

 struct {

      /**
      @memo A pointer to the list of b values
      @doc  Each b value for each SNP
      */
	 unsigned int *b;

      /**
      @memo A pointer to the list of c values
      @doc  Each c value for each SNP
      */
      unsigned int *c;

	  /**
      @memo A pointer to the list of b1 values
      @doc  Each b1 value for each SNP
      */
      unsigned int *b1;

      /**
      @memo A pointer to the list of c values
      @doc  Each c1 value for each SNP
      */
      unsigned int *c1;

	  /**
      @memo A pointer to the list of bmax values
      @doc  Each b value for each SNP
      */
      unsigned int *bmax;

      /**
      @memo A pointer to the list of cmax values
      @doc  Each c value for each SNP
      */
      unsigned int *cmax;
	  
	  /**
      @memo An array of 7 pointers to the values of CompleteTrioFrequencies
      */
      unsigned int *CompleteTrioFrequencies[7];

	   
	  /**
      @memo An array of 17 pointers to the values of MissingTrioFrequencies
      */
      unsigned int *MissingTrioFrequencies[17];

	  /**
      @memo A pointer to the list of x2 for standar TDT
      @doc  Each t value for the x2 value of each SNP
      */
      float *t;


	  /**
      @memo A pointer to the list of p values for standar TDT
      */
      float *p;

	  /**
      @memo A pointer to the list of x2 for tmin in rTDT
      */
      float *tmin;

	   /**
      @memo A pointer to the list of p values for tmin in rTDT
      */
      float *pmin;

	  /**
      @memo A pointer to the list of x2 for tmax in rTDT
      */
      float *tmax;

	   /**
      @memo A pointer to the list of p values for tmax in rTDT
      */
      float *pmax;

 } Counters;  // end structure Counters


        

/***************************************************************/
/***     ASSOCIATED FUNCTIONS     ***/
/*********************************************************************/

/* Head */


/////////////////////////



      /* PUBLIC FUNCTIONS (INTERFACE) */

      public:

		  
	  /**
         @memo Compute TDT 
		 @param b, c
         Time complexity O(TotalSNPs*Size)

      */

	  float GetTDT (unsigned int b, unsigned int c);   

		  
	  /**
         @memo Compute MaxTDT 
		 @param b, c, bmin, bmax, cmin, cmax
         Time complexity O(TotalSNPs*Size)

      */
	  float GetMaxTDT (unsigned int bmin, unsigned int bmax, unsigned int cmin, unsigned int cmax);   

		  
	  /**
         @memo Compute MinTDT for the given sample
		 @param b, c, bmin, bmax, cmin, cmax
         Time complexity O(TotalSNPs*Size)

      */
	  float GetMinTDT (unsigned int bmin, unsigned int bmax, unsigned int cmin, unsigned int cmax);   

	  		  
	  /**
         @memo Compute p value for TDT (x2 1 tail, 1 df)
		 @param x2 value
         Time complexity O(1)

      */
	  float GetPValue (float x2);   
	  

	  
	  /**
         @memo Compute the frequencies for every type of complete trio given a SNP 
		 There are 7 different complete trios (first two genotypes represents parents(unordered) and the third is the offspreing genotype)
		 that can be used by TDT:
		 a1: (11 12 11)
		 a2: (11 12 12)
		 a3: (12 12 11)
		 a4: (12 12 12)
		 a5: (12 12 22)
		 a6: (12 22 12)
		 a7: (12 22 22)
		 @param SNP number
         Time complexity O(Size)

      */

	  void GetCompleteTrioFrequencies ();   

	  
	  /**
         @memo Compute the value of b used in standard TDT given a SNP 
		 The value b is the number of transmission of the wild-type allele from an heterozygote parent,
		 using only complete trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getb ();   

	    
	  /**
         @memo Compute the value of c used in standard TDT given a SNP 
		 The value b is the number of transmission of the mutant allele from an heterozygote parent,
		 using only complete trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getc ();   

	  /**
         @memo Compute the value of t used in standard TDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Gett ();   

	    /**
         @memo Compute the p values of t used in standard TDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getp ();   

	  /**
         @memo Compute the frequencies for every type of missing trio given a SNP 
		 There are 17 different missing trios (first two genotypes represents parents(unordered) and the third is the offspreing genotype)
		 that can be used by rTDT:
		 a1: (00 00 00)
		 a2: (00 00 11)
		 a3: (00 00 12)
		 a4: (00 00 22)
		 a5: (00 11 00)
		 a6: (00 11 11)
		 a7: (00 11 12)
		 a8: (00 12 00)
		 a9: (00 12 11)
		 a10: (00 12 12)
		 a11: (00 12 22)
		 a12: (00 22 00)
		 a13: (00 22 12)
		 a14: (00 22 22)
		 a15: (11 12 00)
		 a16: (12 12 00)
		 a17: (12 22 00)
		 @param SNP number
         Time complexity O(Size)

      */
	  void GetMissingTrioFrequencies ();   


 /**
         @memo Compute the value of b1 used in rTDT given a SNP 
		 The value b1 is the number of known transmissions of the wild-type allele from an heterozygote parent,
		 using only missing trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getb1 ();   

	    
	  /**
         @memo Compute the value of c1 used in rTDT given a SNP 
		 The value b is the number of known transmissions of the mutant allele from an heterozygote parent,
		 using only missing trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getc1 ();   
/**
         @memo Compute the value of bmax used in rTDT given a SNP 
		 The value bmax is the maximum number of transmission of the wild-type allele from an heterozygote parent,
		 compatible with information at missing trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getbmax ();   

	    
	  /**
         @memo Compute the value of cmax used in rTDT given a SNP 
		 The value b is the maximum number of transmissions of the mutant allele from an heterozygote parent,
		 compatible with information at missing trios
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getcmax ();   

	  /**
         @memo Compute the value of tmin used in rTDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Gettmin ();   

	    /**
         @memo Compute the p values of tmin used in rTDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getpmin ();   

	   /**
         @memo Compute the value of tmax used in rTDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Gettmax ();   

	    /**
         @memo Compute the p values of tmax used in rTDT for all the SNPs
		 @param SNP number
         Time complexity O(Size)

      */
	  void Getpmax ();   
      /**
         @memo Write a new text file with TotalSNPs columns and 12 rows.
		 Each row has the following information:
		 Row 1: b
		 Row 2: c
		 Row 3: b1
		 Row 4: c1
		 Row 5: bmax
		 Row 6: cmax
		 Row 7: x2 for standard TDT
		 Row 8: p value for standard TDT
		 Row 9: x2 for tmin in rTDT
		 Row 10: p value for tmin in rTDT
		 Row 11: x2 for tmax in rTDT
		 Row 12: p value for tmax in rTDT
         @param filename: the filename where results will be written
         Time complexity O(TotalSNPs)

      */

	  void WriteResults (char* filename);   
    
	  /**
	 @memo Destructor
	 @doc
           Deallocate memory used by individual.
           Time complexity O(1).

      */
	  ~rTDT ();


	   /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */
	  rTDT(const unsigned int TotalSNPs, const unsigned int Size);


  //	  individual(const phenotype& porigen, const genotype& gorigen, IndCategory ic);

 /**
         @memo Constructor 
         @doc
          Allocate memory an initialize to null.
          Complexity O(1).

      */

		rTDT(const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize); 


		//int gammai(double a,double x,double &gin,double &gim,double &gip);

		//double gamma(double x);


};  // End of class rTDT



/**********************************/
/* DEFINITIONS OF THE FUNCTIONS */
/**********************************/


///////////////////
//// public ////////
///////////////////


/*____________________________________________________________ */

rTDT::rTDT(const unsigned int TotalSNPs, const unsigned int Size): 
individual(TotalSNPs, Size, 1)
{
};


/*____________________________________________________________ */

rTDT::rTDT(const char* filename, const unsigned int InputTotalSNPs, const unsigned int InputSize): 
individual(filename, InputTotalSNPs, InputSize, 1, (IndCategory)1) 
{
};


/*_____________________________________________________________*/

float rTDT::GetTDT (unsigned int b, unsigned int c)
{
float bf=(float)b, cf=(float)c;
 return (((bf-cf)*(bf-cf))/(bf+cf));
};

/*_____________________________________________________________*/

float rTDT::GetMaxTDT (unsigned int b1, unsigned int bmax, unsigned int c1, unsigned int cmax)

{
float Tb1cmax=GetTDT(b1,cmax), Tbmaxc1=GetTDT(bmax,c1);

 if (Tb1cmax>Tbmaxc1)	
	 return Tb1cmax;
 else return Tbmaxc1;
};

/*_____________________________________________________________*/

float rTDT::GetMinTDT (unsigned int b1, unsigned int bmax, unsigned int c1, unsigned int cmax)
{
float Tb1cmax=GetTDT(b1,cmax), Tbmaxc1=GetTDT(bmax,c1);

 if (b1> cmax)
  return (Tb1cmax);
 else if (bmax<c1)
      return (Tbmaxc1);
	  else return (0); 
};

/*_____________________________________________________________*/
	 
float rTDT::GetPValue (float x2)
{

    return(1-gammai(0.5, (double)x2/2));
	return(0);

};
	  
/*_____________________________________________________________*/


void rTDT::GetCompleteTrioFrequencies ()
{

Phenotype *IndPhenotype=TheFirstPhenotype;
Genotype *IndGenotype=TheFirstGenotype, *FatherGenotype, *MotherGenotype;

for (int cont=0;cont<7;cont++)
{
if (Counters.CompleteTrioFrequencies[cont]==NULL)
if ((Counters.CompleteTrioFrequencies[cont]=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 *(Counters.CompleteTrioFrequencies[cont]+SNP)=0;
}

for (int i=0; i<Size; i++)
{
if (IsAChild (IndPhenotype))
{
FatherGenotype=GetGenotype(GetFather(IndPhenotype));
MotherGenotype=GetGenotype(GetMother(IndPhenotype));

for (unsigned int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
// a1: (11 12 11)
if ((IsHomozygous1Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous1(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[0]+SNP))++;


// a2: (11 12 12)
 if ((IsHomozygous1Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[1]+SNP))++;

// a3: (12 12 11)
 if ((IsHeterozygousHeterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous1(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[2]+SNP))++;

// a4: (12 12 12)
 if ((IsHeterozygousHeterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[3]+SNP))++;

// a5: (12 12 22)
  if ((IsHeterozygousHeterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous2(IndGenotype, SNP)))
 (*(Counters.CompleteTrioFrequencies[4]+SNP))++;

// a6: (12 22 12)
 if ((IsHomozygous2Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[5]+SNP))++;

// a7: (12 22 22)
  if ((IsHomozygous2Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous2(IndGenotype, SNP))) 
 (*(Counters.CompleteTrioFrequencies[6]+SNP))++;
} // for each SNP
} // if is a child
IndGenotype=IndGenotype->Next;
IndPhenotype=IndPhenotype->Next;
} // if individual
}

/*_____________________________________________________________*/

void rTDT::Getb ()
{
if (Counters.b==NULL)
if ((Counters.b=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.b+SNP)=0;

 Counters.b[SNP]=(*(Counters.CompleteTrioFrequencies[0]+SNP))+
	 2*(*(Counters.CompleteTrioFrequencies[2]+SNP))+
	 (*(Counters.CompleteTrioFrequencies[3]+SNP))+
	 (*(Counters.CompleteTrioFrequencies[5]+SNP));
}
}

/*_____________________________________________________________*/

void rTDT::Getc ()
{
if (Counters.c==NULL)
if ((Counters.c=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.c+SNP)=0;

 Counters.c[SNP]=(*(Counters.CompleteTrioFrequencies[1]+SNP))+
	 (*(Counters.CompleteTrioFrequencies[3]+SNP))+
	 2*(*(Counters.CompleteTrioFrequencies[4]+SNP))+
	 (*(Counters.CompleteTrioFrequencies[6]+SNP));
}
}


/*_____________________________________________________________*/

void rTDT::Gett ()
{
if (Counters.t==NULL)
if ((Counters.t=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.b==NULL)
 throw NoMemory();
if (Counters.c==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
// *(Counters.t+SNP)=GetTDT((*(Counters.b+SNP)), (*(Counters.c+SNP)));
 Counters.t[SNP]=GetTDT(Counters.b[SNP], Counters.c[SNP]);
}


/*_____________________________________________________________*/

void rTDT::Getp ()
{
if (Counters.p==NULL)
if ((Counters.p=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.t==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 Counters.p[SNP]=GetPValue(Counters.t[SNP]);
}

/*_____________________________________________________________*/


void rTDT::GetMissingTrioFrequencies ()
{

Phenotype *IndPhenotype=TheFirstPhenotype;
Genotype *IndGenotype=TheFirstGenotype, *FatherGenotype, *MotherGenotype;

for (int cont=0;cont<17;cont++)
{
if (Counters.MissingTrioFrequencies[cont]==NULL)
if ((Counters.MissingTrioFrequencies[cont]=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 *(Counters.MissingTrioFrequencies[cont]+SNP)=0;
}


for (int i=0; i<Size; i++)
{
if (IsAChild (IndPhenotype))
{
FatherGenotype=GetGenotype(GetFather(IndPhenotype));
MotherGenotype=GetGenotype(GetMother(IndPhenotype));

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{

// a1: (00 00 00)
 if ((IsMissingMissing(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[0]+SNP))++;


// a2: (00 00 11)
  if ((IsMissingMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous1(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[1]+SNP))++;

// a3: (00 00 12)
  if ((IsMissingMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[2]+SNP))++;

// a4: (00 00 22)
if ((IsMissingMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous2(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[3]+SNP))++;


// a5: (00 11 00)
if ((IsHomozygous1Missing(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[4]+SNP))++;


// a6: (00 11 11)
 if ((IsHomozygous1Missing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous1(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[5]+SNP))++;

// a7: (00 11 12)
 if ((IsHomozygous1Missing(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[6]+SNP))++;

// a8: (00 12 00)
 if ((IsHeterozygousMissing(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[7]+SNP))++;

// a9: (00 12 11)
if ((IsHeterozygousMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous1(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[8]+SNP))++;

// a10: (00 12 12)
if ((IsHeterozygousMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[9]+SNP))++;

// a11: (00 12 22)
if ((IsHeterozygousMissing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous2(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[10]+SNP))++;

// a12: (00 22 00)
if ((IsHomozygous2Missing(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[11]+SNP))++;

// a13: (00 22 12)
if ((IsHomozygous2Missing(FatherGenotype, MotherGenotype, SNP)) && (IsHeterozygous(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[12]+SNP))++;

// a14: (00 22 22)
if ((IsHomozygous2Missing(FatherGenotype, MotherGenotype, SNP)) && (IsHomozygous2(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[13]+SNP))++;

// a15: (11 12 00)
 if ((IsHomozygous1Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[14]+SNP))++;

// a16: (12 12 00)
if ((IsHeterozygousHeterozygous(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[15]+SNP))++;

// a17: (12 22 00)
if ((IsHomozygous2Heterozygous(FatherGenotype, MotherGenotype, SNP)) && (!IsANonMissingSNP(IndGenotype, SNP))) 
 (*(Counters.MissingTrioFrequencies[16]+SNP))++;

} // for each SNP
} // if is a child
IndGenotype=IndGenotype->Next;
IndPhenotype=IndPhenotype->Next;
} // if inidividual
}

/*_____________________________________________________________*/

void rTDT::Getb1 ()
{
if (Counters.b1==NULL)
if ((Counters.b1=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.b==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.b1+SNP)=0;

 Counters.b1[SNP]=Counters.b[SNP]+(*(Counters.MissingTrioFrequencies[8]+SNP));
}
}

/*_____________________________________________________________*/

void rTDT::Getc1 ()
{
if (Counters.c1==NULL)
if ((Counters.c1=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.c==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.c1+SNP)=0;

 Counters.c1[SNP]=Counters.c[SNP]+(*(Counters.MissingTrioFrequencies[10]+SNP));
}
}

/*_____________________________________________________________*/

void rTDT::Getbmax ()
{
if (Counters.bmax==NULL)
if ((Counters.bmax=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.b==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.bmax+SNP)=0;

 Counters.bmax[SNP]=Counters.b[SNP]+
2*(*(Counters.MissingTrioFrequencies[0]+SNP))+
2*(*(Counters.MissingTrioFrequencies[1]+SNP))+
(*(Counters.MissingTrioFrequencies[2]+SNP))+
(*(Counters.MissingTrioFrequencies[4]+SNP))+
(*(Counters.MissingTrioFrequencies[5]+SNP))+
2*(*(Counters.MissingTrioFrequencies[7]+SNP))+
2*(*(Counters.MissingTrioFrequencies[8]+SNP))+
(*(Counters.MissingTrioFrequencies[9]+SNP))+
(*(Counters.MissingTrioFrequencies[11]+SNP))+
(*(Counters.MissingTrioFrequencies[12]+SNP))+
(*(Counters.MissingTrioFrequencies[14]+SNP))+
2*(*(Counters.MissingTrioFrequencies[15]+SNP))+
(*(Counters.MissingTrioFrequencies[16]+SNP));

}
}
/*_____________________________________________________________*/

void rTDT::Getcmax ()
{
if (Counters.cmax==NULL)
if ((Counters.cmax=new unsigned int[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.c==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
{
 *(Counters.cmax+SNP)=0;

 Counters.cmax[SNP]=Counters.c[SNP]+
2*(*(Counters.MissingTrioFrequencies[0]+SNP))+
(*(Counters.MissingTrioFrequencies[2]+SNP))+
2*(*(Counters.MissingTrioFrequencies[3]+SNP))+
(*(Counters.MissingTrioFrequencies[4]+SNP))+
(*(Counters.MissingTrioFrequencies[6]+SNP))+
2*(*(Counters.MissingTrioFrequencies[7]+SNP))+
(*(Counters.MissingTrioFrequencies[9]+SNP))+
2*(*(Counters.MissingTrioFrequencies[10]+SNP))+
(*(Counters.MissingTrioFrequencies[11]+SNP))+
(*(Counters.MissingTrioFrequencies[13]+SNP))+
(*(Counters.MissingTrioFrequencies[14]+SNP))+
2*(*(Counters.MissingTrioFrequencies[15]+SNP))+
(*(Counters.MissingTrioFrequencies[16]+SNP));
}
}
      
/*_____________________________________________________________*/

void rTDT::Gettmin ()
{
if (Counters.tmin==NULL)
if ((Counters.tmin=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.b1==NULL)
 throw NoMemory();
if (Counters.c1==NULL)
 throw NoMemory();
if (Counters.bmax==NULL)
 throw NoMemory();
if (Counters.cmax==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 Counters.tmin[SNP]=GetMinTDT(Counters.b1[SNP], Counters.bmax[SNP], Counters.c1[SNP], Counters.cmax[SNP]);
}

/*_____________________________________________________________*/

void rTDT::Gettmax ()
{
if (Counters.tmax==NULL)
if ((Counters.tmax=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.b1==NULL)
 throw NoMemory();
if (Counters.c1==NULL)
 throw NoMemory();
if (Counters.bmax==NULL)
 throw NoMemory();
if (Counters.cmax==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 Counters.tmax[SNP]=GetMaxTDT(Counters.b1[SNP], Counters.bmax[SNP], Counters.c1[SNP], Counters.cmax[SNP]);
}

/*_____________________________________________________________*/

void rTDT::Getpmin ()
{
if (Counters.pmin==NULL)
if ((Counters.pmin=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.tmin==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 Counters.pmin[SNP]=GetPValue(Counters.tmin[SNP]);
}

/*_____________________________________________________________*/

void rTDT::Getpmax ()
{
if (Counters.pmax==NULL)
if ((Counters.pmax=new float[genotype::TotalSNPs])==NULL)
 throw NoMemory();
if (Counters.tmax==NULL)
 throw NoMemory();

for (int SNP=0;SNP<genotype::TotalSNPs;SNP++)
 Counters.pmax[SNP]=GetPValue(Counters.tmax[SNP]);
}

/*____________________________________________________________ */

void rTDT::WriteResults (char* filename)
{

ofstream OutputFile; 
int SNP;
try
{
 OutputFile.open (filename, ofstream::out);
}
catch (ErrorFile NoFile) {
        NoFile.PrintMessage();
      }


 OutputFile <<  "b:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.b[SNP] <<" ";
  
 OutputFile <<  "\nc:\n";

for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.c[SNP] <<" ";

 OutputFile <<  "\nb1:\n";


for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.b1[SNP] <<" ";

 OutputFile <<  "\nc1\n";

for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.c1[SNP] <<" ";

 OutputFile <<  "\nbmax:\n";

for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.bmax[SNP] <<" ";

 OutputFile <<  "\ncmax:\n";

for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.cmax[SNP] <<" ";

 OutputFile <<  "\nt:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.t[SNP] <<" ";

 OutputFile <<  "\np:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.p[SNP] <<" ";

 OutputFile <<  "\ntmin:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.tmin[SNP] <<" ";

 OutputFile <<  "\npmin:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.pmin[SNP] <<" ";

 OutputFile <<  "\ntmax:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
  OutputFile <<  Counters.tmax[SNP] <<" ";

 OutputFile <<  "\npmax:\n";

 for (SNP=0; SNP<genotype::TotalSNPs;SNP++)
 OutputFile <<  Counters.pmax[SNP] <<" ";

 OutputFile <<  "\n";

 OutputFile.close();
}


/*_______________________________________________________________________*/

rTDT::~rTDT ()
{
	

       		assert (Counters.b != NULL);
			delete Counters.b;
			assert (Counters.c != NULL);
			delete Counters.c;
			assert (Counters.t != NULL);
			delete Counters.t;
			assert (Counters.p != NULL);
			delete Counters.p;
			assert (Counters.b1 != NULL);
			delete Counters.c1;
			assert (Counters.bmax != NULL);
			delete Counters.bmax;
			assert (Counters.c1 != NULL);
			delete Counters.c1;
			assert (Counters.cmax != NULL);
			delete Counters.cmax;
			assert (Counters.tmin != NULL);
			delete Counters.tmin;
			assert (Counters.tmax != NULL);
			delete Counters.tmax;
			assert (Counters.pmin != NULL);
			delete Counters.pmin;
			assert (Counters.pmax != NULL);
			delete Counters.pmax;          
            
}


/*____________________________________________________________ */




};  // Fin del Namespace

//#endif

/* Fin Fichero: rTDT.h */
